
'use server';
/**
 * @fileOverview A Genkit flow to analyze the sentiment of a blog comment.
 *
 * - analyzeCommentSentiment - A function that takes comment text and returns its sentiment.
 * - AnalyzeCommentSentimentInput - The input type for the analyzeCommentSentiment function.
 * - AnalyzeCommentSentimentOutput - The return type for the analyzeCommentSentiment function.
 * - CommentSentiment - The type for possible sentiment values.
 */

import { ai } from '@/ai/genkit';
import { z } from 'genkit';
import type { AnalyzeCommentSentimentInput, AnalyzeCommentSentimentOutput, CommentSentiment } from '@/lib/data';

// Zod schema for sentiment values
const CommentSentimentSchema = z.enum(["positive", "negative", "neutral"]);

// Zod schema for the flow input
const AnalyzeCommentSentimentInputSchema = z.object({
  commentText: z.string().min(1).describe('The text of the comment to analyze.'),
});

// Zod schema for the flow output
const AnalyzeCommentSentimentOutputSchema = z.object({
  sentiment: CommentSentimentSchema.describe('The determined sentiment of the comment (positive, negative, or neutral).'),
  confidence: z.number().min(0).max(1).optional().describe('The confidence score (0.0 to 1.0) of the sentiment analysis, if available.'),
});

// Export the wrapper function that calls the flow
export async function analyzeCommentSentiment(input: AnalyzeCommentSentimentInput): Promise<AnalyzeCommentSentimentOutput> {
  return analyzeCommentSentimentFlow(input);
}

const analyzeCommentSentimentPrompt = ai.definePrompt({
  name: 'analyzeCommentSentimentPrompt',
  input: { schema: AnalyzeCommentSentimentInputSchema },
  output: { schema: AnalyzeCommentSentimentOutputSchema },
  prompt: `Analyze the sentiment of the following comment text.
Classify it as "positive", "negative", or "neutral".
If possible, provide a confidence score between 0.0 and 1.0 for your classification.

Comment Text:
{{{commentText}}}

Return your analysis in JSON format matching the output schema.
Focus on the overall feeling conveyed by the comment.
If the comment is very short or ambiguous, lean towards "neutral" unless strong sentiment indicators are present.
`,
});

const analyzeCommentSentimentFlow = ai.defineFlow(
  {
    name: 'analyzeCommentSentimentFlow',
    inputSchema: AnalyzeCommentSentimentInputSchema,
    outputSchema: AnalyzeCommentSentimentOutputSchema,
  },
  async (input) => {
    const { output } = await analyzeCommentSentimentPrompt(input);
    
    if (!output || !output.sentiment) {
      // Fallback if the LLM fails to provide a valid sentiment
      console.warn("Sentiment analysis flow did not return a valid sentiment. Defaulting to neutral.", output);
      return { sentiment: "neutral" };
    }
    return output;
  }
);

// Exporting types for external use
export type { AnalyzeCommentSentimentInput, AnalyzeCommentSentimentOutput, CommentSentiment };
