
'use server';
/**
 * @fileOverview A Genkit flow to suggest related blog posts.
 *
 * - suggestRelatedPosts - A function that takes details of a current post and a list of all posts,
 *   and returns a list of suggested related posts.
 * - SuggestRelatedPostsInput - The input type for the suggestRelatedPosts function.
 * - SuggestRelatedPostsOutput - The return type for the suggestRelatedPosts function.
 */

import { ai } from '@/ai/genkit';
import { z } from 'genkit';
import type { SuggestRelatedPostsInput, SuggestRelatedPostsOutput, SimplePostInfo, SuggestedPost } from '@/lib/data'; // Assuming types are in lib/data

// Define Zod schemas based on the TypeScript interfaces
const SimplePostInfoSchema = z.object({
  title: z.string().describe('The title of the blog post.'),
  slug: z.string().describe('The URL slug of the blog post.'),
  tags: z.array(z.string()).describe('A list of tags associated with the post.'),
  category: z.string().describe('The main category of the post.'),
  excerpt: z.string().describe('A short summary or excerpt of the post.'),
});

const SuggestRelatedPostsInputSchema = z.object({
  currentPostTitle: z.string().describe('The title of the current blog post for which to find related articles.'),
  currentPostTags: z.array(z.string()).describe('The tags of the current blog post.'),
  currentPostCategory: z.string().describe('The category of the current blog post.'),
  currentPostExcerpt: z.string().describe('An excerpt from the current blog post. Can be of any length.'), // Removed .min(50)
  allPosts: z.array(SimplePostInfoSchema).describe('A list of all available blog posts, excluding the current one, to choose related articles from. Each post includes title, slug, tags, category, and excerpt.'),
});

const SuggestedPostSchema = z.object({
  title: z.string().describe('The title of the suggested related post.'),
  slug: z.string().describe('The URL slug of the suggested related post.'),
});

const SuggestRelatedPostsOutputSchema = z.object({
  relatedPosts: z.array(SuggestedPostSchema).max(3).describe('A list of up to 3 suggested related blog posts, each with a title and slug.'),
});


// Export the wrapper function that calls the flow
export async function suggestRelatedPosts(input: SuggestRelatedPostsInput): Promise<SuggestRelatedPostsOutput> {
  return suggestRelatedPostsFlow(input);
}

const suggestRelatedPostsPrompt = ai.definePrompt({
  name: 'suggestRelatedPostsPrompt',
  input: { schema: SuggestRelatedPostsInputSchema },
  output: { schema: SuggestRelatedPostsOutputSchema },
  prompt: `You are an expert blog content curator. Your task is to suggest relevant articles.
Given the details of a current blog post:
- Title: {{{currentPostTitle}}}
- Tags: {{#each currentPostTags}}{{{this}}}{{#unless @last}}, {{/unless}}{{/each}}
- Category: {{{currentPostCategory}}}
- Excerpt: {{{currentPostExcerpt}}}

And a list of other available blog posts (JSON format):
{{{jsonEncode allPosts}}}

Please analyze the current post and select up to 3 most relevant articles from the provided list.
Focus on thematic similarity, shared tags, common categories, and content overlap indicated by the excerpts.
Do not suggest the current post itself if it accidentally appears in the 'allPosts' list.

Return your suggestions as a JSON object matching the output schema.
The output should be a list named 'relatedPosts', containing objects with 'title' and 'slug' for each suggested article.
If no suitable related posts are found from the list, return an empty 'relatedPosts' array.
`,
  // Helper for JSON encoding to be used in the prompt
  // Note: Genkit's built-in Handlebars might not have jsonEncode.
  // If this doesn't work, the `allPosts` array might need to be stringified before being passed to the prompt,
  // or the prompt needs to be adjusted to work with the array directly if Handlebars supports complex object iteration well.
  // For simplicity, assuming `jsonEncode` helper or similar functionality.
  // If not, stringify in the flow: `JSON.stringify(input.allPosts)`
});

const suggestRelatedPostsFlow = ai.defineFlow(
  {
    name: 'suggestRelatedPostsFlow',
    inputSchema: SuggestRelatedPostsInputSchema,
    outputSchema: SuggestRelatedPostsOutputSchema,
  },
  async (input) => {
    // Pre-process input if needed, e.g., stringify allPosts if jsonEncode isn't available in prompt
    const processedInput = {
      ...input,
      // Stringify allPosts to ensure it's passed as a valid JSON string in the prompt
      // The prompt then expects to see this string and parse it (implicitly by the model or explicitly if needed)
      // However, the model is generally good at understanding structured data if described.
      // Let's assume the model can handle the object directly with the provided schema and prompt.
      // If issues arise, stringifying might be a fallback:
      // allPosts: JSON.stringify(input.allPosts),
    };

    const { output } = await suggestRelatedPostsPrompt(processedInput);
    
    if (!output || !output.relatedPosts) {
      // Ensure we always return the correct structure, even if the LLM fails to.
      return { relatedPosts: [] };
    }
    return output;
  }
);

// Exporting types for external use (e.g., in page components)
export type { SuggestRelatedPostsInput, SuggestRelatedPostsOutput, SimplePostInfo, SuggestedPost };
