
import type { ReactNode } from 'react';
import { Header } from '@/components/layout/Header';
import { Footer } from '@/components/layout/Footer';
import { BlogSidebar } from '@/components/blog/BlogSidebar';
import {
  type BlogCategory, type BlogPost, type NavigationLink, type SocialLink, type BlogTag,
  defaultSocialLinks, getIcon
} from '@/lib/data';
import {
  getBlogCategoriesFromApi, getBlogPosts, getSocialLinks, getBlogTagsFromApi
} from '@/services/apiService';
import { personalInfo } from '@/lib/data'; // For metadata
import { Home, Rss } from 'lucide-react';

export async function generateMetadata() {
  return {
    title: `Blog | ${personalInfo.name}`,
    description: `Insights, articles, and updates from ${personalInfo.name}.`,
  };
}

export default async function BlogLayout({ children }: { children: ReactNode }) {
  let categories: BlogCategory[] = [];
  let recentPostsForSidebar: BlogPost[] = [];
  let tagsForSidebar: BlogTag[] = [];
  let socialLinksData: SocialLink[] = defaultSocialLinks;

  let categoryError: string | null = null;
  let postsError: string | null = null;
  let tagsError: string | null = null;
  let socialLinksError: string | null = null;
  
  let isLoadingSocialLinksForLayout = true;

  const blogNavigationLinks: NavigationLink[] = [
    { name: "Back to Resume", href: "/", iconName: "Home" },
    { name: "All Posts", href: "/blog", iconName: "Rss" },
  ];

  // Fetch Social Links
  const socialLinksResponse = await getSocialLinks();
  if (socialLinksResponse.success && socialLinksResponse.data && socialLinksResponse.data.length > 0) {
    socialLinksData = socialLinksResponse.data;
  } else {
    socialLinksError = socialLinksResponse.error || "Failed to load social links. Using defaults.";
    console.warn("BlogLayout: Social links API failed or no data. " + socialLinksError);
    // socialLinksData remains defaultSocialLinks
  }
  isLoadingSocialLinksForLayout = false;

  // Fetch Categories from API
  const apiCategoriesResponse = await getBlogCategoriesFromApi();
  if (apiCategoriesResponse.success && apiCategoriesResponse.data && apiCategoriesResponse.data.length > 0) {
    categories = apiCategoriesResponse.data;
    console.log("BlogLayout: Successfully fetched categories from API:", categories.length, "items.");
  } else {
    categoryError = apiCategoriesResponse.error || "API call for categories failed or no categories found.";
    console.warn("BlogLayout: " + categoryError + ". Categories for sidebar will be empty.");
    categories = []; 
  }

  // Fetch Tags from API
  const apiTagsResponse = await getBlogTagsFromApi();
  if (apiTagsResponse.success && apiTagsResponse.data && apiTagsResponse.data.length > 0) {
    tagsForSidebar = apiTagsResponse.data.sort((a, b) => (b.count || 0) - (a.count || 0));
     console.log("BlogLayout: Successfully fetched tags from API:", tagsForSidebar.length, "items.");
  } else {
    tagsError = apiTagsResponse.error || "API call for tags failed or no tags found.";
    console.warn("BlogLayout: " + tagsError + ". Tags for sidebar will be empty or derived if posts load.");
    tagsForSidebar = [];
  }
  
  // Fetch Blog Posts for Sidebar (Recent Posts) and potentially for deriving tags if API fails for tags
  const blogPostsResponse = await getBlogPosts({ filters: { limit: 5 } }); 
  if (blogPostsResponse.success && blogPostsResponse.data) {
    recentPostsForSidebar = blogPostsResponse.data;
    console.log("BlogLayout: Successfully fetched recent posts for sidebar from API:", recentPostsForSidebar.length, "items.");

    // Fallback: If tags API failed but posts loaded, derive tags from these posts
    if (tagsForSidebar.length === 0 && apiTagsResponse.error) { // Only derive if tag API explicitly failed
        const tagCounts: Record<string, { name: string, slug: string, count: number }> = {};
        // Consider fetching all posts if a comprehensive tag list is desired for fallback
        // For now, using recentPostsForSidebar for tag derivation to limit API calls.
        // const allPostsForTagDerivation = await getBlogPosts(); 
        // if(allPostsForTagDerivation.success && allPostsForTagDerivation.data) {
        recentPostsForSidebar.forEach(post => { // Deriving from recentPosts for simplicity here
          post.frontmatter.tags?.forEach(tagName => { 
            const slug = tagName.toLowerCase().replace(/\s+/g, '-');
            if (!tagCounts[tagName]) {
                tagCounts[tagName] = { name: tagName, slug: slug, count: 0 };
            }
            tagCounts[tagName].count++;
          });
        });
        tagsForSidebar = Object.values(tagCounts).sort((a, b) => b.count - a.count);
        if (tagsForSidebar.length > 0) {
            console.log("BlogLayout: Derived tags from posts as API for tags failed or returned empty. Count:", tagsForSidebar.length);
            tagsError = null; 
        } else if (!tagsError) { 
            tagsError = "No tags found from API or posts.";
        }
        // } else if (!tagsError) {
        //     tagsError = allPostsForTagDerivation.error || "Could not fetch posts to derive tags.";
        // }
    }


    if (recentPostsForSidebar.length === 0 && !blogPostsResponse.error) {
        postsError = "No recent blog posts available from API.";
    } else if (blogPostsResponse.error && !recentPostsForSidebar.length) {
        postsError = blogPostsResponse.error; // API error and no posts
    } else if (blogPostsResponse.error) { // API error but some posts might still be shown (if any)
        postsError = blogPostsResponse.error; // Log API error even if some posts are present
    }
  } else {
    postsError = blogPostsResponse.error || "API call for recent posts failed.";
    console.warn("BlogLayout: " + postsError + ". Recent posts for sidebar will be empty.");
    recentPostsForSidebar = [];
    if(tagsForSidebar.length === 0 && !tagsError) { // If tags also failed to load from API
        tagsError = tagsError || "Could not load posts to derive tags.";
    }
  }

  return (
    <div className="flex flex-col min-h-screen bg-background text-foreground">
      <Header
        navigationLinks={blogNavigationLinks} // Specific links for blog header
        isLoading={false} // Blog header links are static, not API-driven
        error={null} 
      />
      <main className="flex-grow container mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="lg:flex lg:space-x-8">
          <div className="lg:w-3/4">
            {children}
          </div>
          <aside className="lg:w-1/4 mt-8 lg:mt-0">
            {/* Display errors at the top of the sidebar for visibility */}
            {categoryError && (!categories || categories.length === 0) && <p className="text-xs text-destructive mb-2">Sidebar category loading: {categoryError}</p>}
            {tagsError && (!tagsForSidebar || tagsForSidebar.length === 0) && <p className="text-xs text-destructive mb-2">Sidebar tags loading: {tagsError}</p>}
            {postsError && (!recentPostsForSidebar || recentPostsForSidebar.length === 0) && <p className="text-xs text-destructive mb-2">Sidebar posts loading: {postsError}</p>}
            <BlogSidebar
              recentPosts={recentPostsForSidebar}
              categories={categories}
              tags={tagsForSidebar} // Pass fetched or derived tags
              socialLinks={socialLinksData}
              isLoadingSocialLinks={isLoadingSocialLinksForLayout}
              socialLinksError={socialLinksError}
            />
          </aside>
        </div>
      </main>
      <Footer
        socialLinks={socialLinksData}
        isLoading={isLoadingSocialLinksForLayout}
        error={socialLinksError}
      />
    </div>
  );
}
