
import type { Metadata, Viewport } from 'next';
import { Inter } from 'next/font/google';
import './globals.css';
import { ClientLayoutWrapper } from '@/components/layout/ClientLayoutWrapper';
import { personalInfo, type AppSettings } from '@/lib/data';
import { getAppSettings } from '@/services/apiService';
import Script from 'next/script';

const inter = Inter({
  subsets: ['latin'],
  variable: '--font-inter',
});

export async function generateViewport(): Promise<Viewport> {
  const settingsResponse = await getAppSettings();
  const settings = settingsResponse.success && settingsResponse.data ? settingsResponse.data : null;
  // console.log("[RootLayout - generateViewport] Fetched App Settings:", settings);
  return {
    themeColor: [
      { media: '(prefers-color-scheme: light)', color: 'white' },
      { media: '(prefers-color-scheme: dark)', color: 'black' },
    ],
  };
}

export async function generateMetadata(): Promise<Metadata> {
  const settingsResponse = await getAppSettings();
  const settings = settingsResponse.success && settingsResponse.data ? settingsResponse.data : null;
  // console.log("[RootLayout - generateMetadata] Fetched App Settings:", settings);

  const siteTitle = settings?.title || `${personalInfo.name} | ${personalInfo.title}`;
  const siteDescription = settings?.description || personalInfo.about;
  const siteName = settings?.organization_name || `${personalInfo.name}'s Resume & Blog`;
  const siteUrl = personalInfo.siteUrl;

  return {
    metadataBase: new URL(siteUrl),
    title: {
      default: siteTitle,
      template: `%s | ${settings?.organization_name || personalInfo.name}`,
    },
    description: siteDescription,
    applicationName: siteName,
    authors: [{ name: personalInfo.name, url: siteUrl }],
    generator: 'Next.js',
    keywords: settings?.meta_keywords || "Hussain Abuhajajj, Software Project Manager, Project Management, Full Stack Developer, Laravel, Vue.js, Agile, Scrum, Blog, Tech Insights, Product Reviews, Career Tips",
    referrer: 'origin-when-cross-origin',
    creator: personalInfo.name,
    publisher: settings?.organization_name || personalInfo.name,
    robots: {
      index: true,
      follow: true,
      googleBot: {
        index: true,
        follow: true,
        'max-video-preview': -1,
        'max-image-preview': 'large',
        'max-snippet': -1,
      },
    },
    openGraph: {
      title: siteTitle,
      description: siteDescription,
      url: siteUrl,
      siteName: siteName,
      images: [
        {
          url: settings?.logo_url || personalInfo.defaultOgImage,
          width: 1200,
          height: 630,
          alt: `${siteName} Logo`,
        },
      ],
      locale: 'en_US',
      type: 'profile',
    },
    twitter: {
      card: 'summary_large_image',
      title: siteTitle,
      description: siteDescription,
      creator: '@hussainabuhajaj',
      images: [settings?.logo_url || personalInfo.defaultOgImage],
    },
    verification: {
      google: settings?.google_console_code || undefined,
    },
    alternates: {
      canonical: siteUrl,
    },
    icons: {
      icon: settings?.favicon_url || "/favicon.ico",
    },
  };
}

const personJsonLd = {
  "@context": "https://schema.org",
  "@type": "Person",
  "name": "Hussain Abuhajajj",
  "jobTitle": personalInfo.title,
  "email": `mailto:${personalInfo.email}`,
  "telephone": personalInfo.phone,
  "url": personalInfo.siteUrl,
  "image": personalInfo.avatarImage,
  "sameAs": [
    personalInfo.social.linkedin,
    personalInfo.social.github,
    personalInfo.social.twitter
  ],
  "alumniOf": {
    "@type": "CollegeOrUniversity",
    "name": "College of Intermediate Studies, Al-Azhar University"
  },
  "worksFor": {
    "@type": "Organization",
    "name": "Avccode",
    "startDate": "2016",
    "endDate": "2024"
  },
  "description": personalInfo.about,
  "knowsLanguage": ["Arabic", "English"],
};

export default async function RootLayout({
  children,
}: Readonly<{
  children: React.ReactNode;
}>) {
  const settingsResponse = await getAppSettings();
  const settings = settingsResponse.success && settingsResponse.data ? settingsResponse.data : null;

  console.log("RootLayout: Fetched App Settings:", JSON.stringify(settings, null, 2));
  if (!settingsResponse.success) {
    console.warn("RootLayout: Failed to fetch app settings from API. Error:", settingsResponse.error);
  }

  const gaTrackingId = settings?.google_analytic_code || undefined;
  const adsenseClientId = settings?.google_adsense_code || undefined;
  const gtmId = settings?.google_tag_manager_code || "GTM-TMSC24Q7"; // Fallback GTM ID

  console.log("RootLayout: GA ID:", gaTrackingId, "AdSense ID:", adsenseClientId, "GTM ID:", gtmId);


  return (
    <html lang="en" suppressHydrationWarning>
      <head>
        {gtmId && (
          <script
            dangerouslySetInnerHTML={{
              __html: `(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
})(window,document,'script','dataLayer','${gtmId}');`,
            }}
          />
        )}
        {adsenseClientId && (
          <Script
            id="adsense-script"
            async
            src={`https://pagead2.googlesyndication.com/pagead/js/adsbygoogle.js?client=${adsenseClientId}`}
            crossOrigin="anonymous"
            strategy="afterInteractive"
          />
        )}
        <script
          type="application/ld+json"
          dangerouslySetInnerHTML={{ __html: JSON.stringify(personJsonLd) }}
          key="person-jsonld"
        />
      </head>
      <body className={`${inter.variable} font-sans antialiased`}>
        {gtmId && (
          <noscript>
            <iframe
              src={`https://www.googletagmanager.com/ns.html?id=${gtmId}`}
              height="0"
              width="0"
              style={{ display: 'none', visibility: 'hidden' }}
            ></iframe>
          </noscript>
        )}
        <ClientLayoutWrapper gaTrackingId={gaTrackingId}>
          {children}
        </ClientLayoutWrapper>
      </body>
    </html>
  );
}
