
"use client";

import { useRef, useState, useEffect } from "react";
import { useReactToPrint } from "react-to-print";
import { Header } from "@/components/layout/Header";
import { Footer } from "@/components/layout/Footer";
import { HeroSection } from "@/components/sections/HeroSection";
import { AboutSection } from "@/components/sections/AboutSection";
import { TechStackSection } from "@/components/sections/TechStackSection";
import { ExperienceSection } from "@/components/sections/ExperienceSection";
import { ProjectsSection } from "@/components/sections/ProjectsSection";
import { EducationSection } from "@/components/sections/EducationSection";
import { InterestsSection } from "@/components/sections/InterestsSection";
import { TestimonialsSection } from "@/components/sections/TestimonialsSection";
import { PatreonSupportSection } from "@/components/sections/PatreonSupportSection";
import { ContactSection } from "@/components/sections/ContactSection";
import { LatestBlogPostsSection } from "@/components/sections/LatestBlogPostsSection";
import { FeaturedBlogPostsSection } from "@/components/sections/FeaturedBlogPostsSection";
import { AdSensePlaceholder } from "@/components/AdSensePlaceholder"; // Import reusable component
import {
  personalInfo,
  type EducationEntry, type Project, type BlogPost,
  type TechStackCategory, type ExperienceEntry, type Testimonial, type PatreonTier, type Interest,
  type NavigationLink, type SocialLink,
  defaultNavigationLinks, defaultSocialLinks
} from "@/lib/data";
import {
  getEducationItems, getProjectsFromApi, getBlogPosts,
  getTechStack, getExperience, getTestimonials, getPatreonTiers, getInterests,
  getNavigationLinks, getSocialLinks
} from "@/services/apiService";

export default function InteractiveResumePage() {
  const resumeContentRef = useRef<HTMLDivElement>(null);

  const [educationData, setEducationData] = useState<EducationEntry[] | null>(null);
  const [isLoadingEducation, setIsLoadingEducation] = useState(true);
  const [educationError, setEducationError] = useState<string | null>(null);

  const [projectsData, setProjectsData] = useState<Project[] | null>(null);
  const [isLoadingProjects, setIsLoadingProjects] = useState(true);
  const [projectsError, setProjectsError] = useState<string | null>(null);

  const [recentPosts, setRecentPosts] = useState<BlogPost[] | null>(null);
  const [isLoadingRecentPosts, setIsLoadingRecentPosts] = useState(true);
  const [recentPostsError, setRecentPostsError] = useState<string | null>(null);

  const [featuredPosts, setFeaturedPosts] = useState<BlogPost[] | null>(null);
  const [isLoadingFeaturedPosts, setIsLoadingFeaturedPosts] = useState(true);
  const [featuredPostsError, setFeaturedPostsError] = useState<string | null>(null);

  const [techStackData, setTechStackData] = useState<TechStackCategory[] | null>(null);
  const [isLoadingTechStack, setIsLoadingTechStack] = useState(true);
  const [techStackError, setTechStackError] = useState<string | null>(null);

  const [experienceData, setExperienceData] = useState<ExperienceEntry[] | null>(null);
  const [isLoadingExperience, setIsLoadingExperience] = useState(true);
  const [experienceError, setExperienceError] = useState<string | null>(null);

  const [testimonialsData, setTestimonialsData] = useState<Testimonial[] | null>(null);
  const [isLoadingTestimonials, setIsLoadingTestimonials] = useState(true);
  const [testimonialsError, setTestimonialsError] = useState<string | null>(null);

  const [patreonTiersData, setPatreonTiersData] = useState<PatreonTier[] | null>(null);
  const [isLoadingPatreonTiers, setIsLoadingPatreonTiers] = useState(true);
  const [patreonTiersError, setPatreonTiersError] = useState<string | null>(null);

  const [interestsData, setInterestsData] = useState<Interest[] | null>(null);
  const [isLoadingInterests, setIsLoadingInterests] = useState(true);
  const [interestsError, setInterestsError] = useState<string | null>(null);

  const [navigationLinksData, setNavigationLinksData] = useState<NavigationLink[]>(defaultNavigationLinks);
  const [isLoadingNavigationLinks, setIsLoadingNavigationLinks] = useState(true);
  const [navigationLinksError, setNavigationLinksError] = useState<string | null>(null);

  const [socialLinksData, setSocialLinksData] = useState<SocialLink[]>(defaultSocialLinks);
  const [isLoadingSocialLinks, setIsLoadingSocialLinks] = useState(true);
  const [socialLinksError, setSocialLinksError] = useState<string | null>(null);

  useEffect(() => {
    async function fetchData() {
      setIsLoadingEducation(true); setEducationError(null);
      setIsLoadingProjects(true); setProjectsError(null);
      setIsLoadingRecentPosts(true); setRecentPostsError(null);
      setIsLoadingFeaturedPosts(true); setFeaturedPostsError(null);
      setIsLoadingTechStack(true); setTechStackError(null);
      setIsLoadingExperience(true); setExperienceError(null);
      setIsLoadingTestimonials(true); setTestimonialsError(null);
      setIsLoadingPatreonTiers(true); setPatreonTiersError(null);
      setIsLoadingInterests(true); setInterestsError(null);
      setIsLoadingNavigationLinks(true); setNavigationLinksError(null);
      setIsLoadingSocialLinks(true); setSocialLinksError(null);

      try {
        // Education
        const eduResponse = await getEducationItems();
        if (eduResponse.success) {
          setEducationData(eduResponse.data ?? []);
          if (!eduResponse.data || eduResponse.data.length === 0) {
            setEducationError("No education details available from API.");
          }
        } else {
          setEducationError(eduResponse.error || "Failed to load education details. Displaying default.");
          setEducationData([]); // Fallback to empty or could use static here if preferred for errors
        }

        // Projects
        const projResponse = await getProjectsFromApi();
        if (projResponse.success) {
          setProjectsData(projResponse.data ?? []);
           if (!projResponse.data || projResponse.data.length === 0) {
            setProjectsError("No projects available from API.");
          }
        } else {
          setProjectsError(projResponse.error || "Failed to load projects. Displaying default.");
          setProjectsData([]);
        }

        // Blog Posts (Recent & Featured)
        const blogPostsResponse = await getBlogPosts(); // Fetches all published posts
        if (blogPostsResponse.success && Array.isArray(blogPostsResponse.data)) {
            const allApiPosts = blogPostsResponse.data.filter(post => post && post.slug && post.frontmatter && post.frontmatter.status === 'PUBLISHED');

            const recent = allApiPosts.slice(0, 3);
            setRecentPosts(recent);
            if (recent.length === 0) {
                setRecentPostsError("No recent blog posts available.");
            }

            const featured = allApiPosts.filter(p => p.frontmatter.featured).slice(0, 3);
            setFeaturedPosts(featured);
            if (featured.length === 0) {
                setFeaturedPostsError("No featured blog posts available.");
            }
        } else {
            const commonBlogError = blogPostsResponse.error || "Could not load blog posts.";
            setRecentPostsError(commonBlogError);
            setFeaturedPostsError(commonBlogError);
            setRecentPosts([]);
            setFeaturedPosts([]);
        }


        // Tech Stack
        const techStackResponse = await getTechStack();
        if (techStackResponse.success) {
          setTechStackData(techStackResponse.data ?? []);
          if (!techStackResponse.data || techStackResponse.data.length === 0) {
            setTechStackError("No tech stack details available from API.");
          }
        } else {
          setTechStackError(techStackResponse.error || "Failed to load tech stack.");
          setTechStackData([]);
        }

        // Experience
        const experienceResponse = await getExperience();
        if (experienceResponse.success) {
          setExperienceData(experienceResponse.data ?? []);
          if (!experienceResponse.data || experienceResponse.data.length === 0) {
            setExperienceError("No experience details available from API.");
          }
        } else {
          setExperienceError(experienceResponse.error || "Failed to load experience.");
          setExperienceData([]);
        }

        // Testimonials
        const testimonialsResponse = await getTestimonials();
        if (testimonialsResponse.success) {
          setTestimonialsData(testimonialsResponse.data ?? []);
          if (!testimonialsResponse.data || testimonialsResponse.data.length === 0) {
            setTestimonialsError("No testimonials available from API.");
          }
        } else {
          setTestimonialsError(testimonialsResponse.error || "Failed to load testimonials.");
          setTestimonialsData([]);
        }
        
        // Patreon Tiers
        const patreonTiersResponse = await getPatreonTiers();
        if (patreonTiersResponse.success) {
          setPatreonTiersData(patreonTiersResponse.data ?? []);
          if (!patreonTiersResponse.data || patreonTiersResponse.data.length === 0) {
            setPatreonTiersError("No Patreon tiers available from API.");
          }
        } else {
          setPatreonTiersError(patreonTiersResponse.error || "Failed to load Patreon tiers.");
          setPatreonTiersData([]);
        }

        // Interests
        const interestsResponse = await getInterests();
        if (interestsResponse.success) {
          setInterestsData(interestsResponse.data ?? []);
          if (!interestsResponse.data || interestsResponse.data.length === 0) {
            setInterestsError("No interests available from API.");
          }
        } else {
          setInterestsError(interestsResponse.error || "Failed to load interests.");
          setInterestsData([]);
        }
        
        // Navigation Links
        const navLinksResponse = await getNavigationLinks();
        if (navLinksResponse.success && navLinksResponse.data && navLinksResponse.data.length > 0) {
          setNavigationLinksData(navLinksResponse.data);
        } else {
          setNavigationLinksData(defaultNavigationLinks); // Retain default if API fails or no data
          setNavigationLinksError(navLinksResponse.error || "Navigation links not found, using defaults.");
        }

        // Social Links
        const socialLinksResponse = await getSocialLinks();
        if (socialLinksResponse.success && socialLinksResponse.data && socialLinksResponse.data.length > 0) {
          setSocialLinksData(socialLinksResponse.data);
        } else {
          setSocialLinksData(defaultSocialLinks); // Retain default
          setSocialLinksError(socialLinksResponse.error || "Social links not found, using defaults.");
        }

      } catch (err) {
        console.error("Global error in fetchData:", err);
        const genericError = "A general error occurred while fetching page data.";
        setEducationError(educationError || genericError);
        setProjectsError(projectsError || genericError);
        setRecentPostsError(recentPostsError || genericError);
        setFeaturedPostsError(featuredPostsError || genericError);
        setTechStackError(techStackError || genericError);
        setExperienceError(experienceError || genericError);
        setTestimonialsError(testimonialsError || genericError);
        setPatreonTiersError(patreonTiersError || genericError);
        setInterestsError(interestsError || genericError);
        if (navigationLinksData.length === defaultNavigationLinks.length && navigationLinksData[0].name === defaultNavigationLinks[0].name) { // Avoid overwriting if API already populated
           setNavigationLinksError(navigationLinksError || "Failed to load navigation. Using defaults.");
        }
        if (socialLinksData.length === defaultSocialLinks.length && socialLinksData[0].name === defaultSocialLinks[0].name) {
            setSocialLinksError(socialLinksError || "Failed to load social links. Using defaults.");
        }
      } finally {
        setIsLoadingEducation(false);
        setIsLoadingProjects(false);
        setIsLoadingRecentPosts(false);
        setIsLoadingFeaturedPosts(false);
        setIsLoadingTechStack(false);
        setIsLoadingExperience(false);
        setIsLoadingTestimonials(false);
        setIsLoadingPatreonTiers(false);
        setIsLoadingInterests(false);
        setIsLoadingNavigationLinks(false);
        setIsLoadingSocialLinks(false);
      }
    }
    fetchData();
  // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const handlePrint = useReactToPrint({
    content: () => resumeContentRef.current,
    documentTitle: `${personalInfo.name.replace(/\s+/g, "_")}_Resume`,
    onBeforeGetContent: () => Promise.resolve(),
    onAfterPrint: () => {},
    removeAfterPrint: true,
  });

  return (
    <div className="flex flex-col min-h-screen bg-background text-foreground print:h-auto">
      <Header
        onDownloadResume={handlePrint}
        navigationLinks={navigationLinksData}
        isLoading={isLoadingNavigationLinks}
        error={navigationLinksError}
      />
      <main
        ref={resumeContentRef}
        className="flex-grow overflow-y-auto scroll-smooth no-print"
      >
        <HeroSection />
        <AboutSection />
        <AdSensePlaceholder slot="Homepage - After About" />
        <TechStackSection
          techStackCategories={techStackData}
          isLoading={isLoadingTechStack}
          error={techStackError}
        />
        <ExperienceSection
          experienceEntries={experienceData}
          isLoading={isLoadingExperience}
          error={experienceError}
        />
        <AdSensePlaceholder slot="Homepage - After Experience" />
        <EducationSection
          educationItems={educationData}
          isLoading={isLoadingEducation}
          error={educationError}
        />
        <ProjectsSection
          projects={projectsData}
          isLoading={isLoadingProjects}
          error={projectsError}
        />
        <AdSensePlaceholder slot="Homepage - After Projects" />
        <LatestBlogPostsSection
          posts={recentPosts}
          isLoading={isLoadingRecentPosts}
          error={recentPostsError}
        />
        <FeaturedBlogPostsSection
          posts={featuredPosts}
          isLoading={isLoadingFeaturedPosts}
          error={featuredPostsError}
        />
        <InterestsSection
          interests={interestsData}
          isLoading={isLoadingInterests}
          error={interestsError}
        />
        <TestimonialsSection
          testimonials={testimonialsData}
          isLoading={isLoadingTestimonials}
          error={testimonialsError}
        />
        <PatreonSupportSection
          patreonTiers={patreonTiersData}
          isLoading={isLoadingPatreonTiers}
          error={patreonTiersError}
        />
        <ContactSection />
      </main>
      <Footer
        socialLinks={socialLinksData}
        isLoading={isLoadingSocialLinks}
        error={socialLinksError}
      />
    </div>
  );
}
