
"use client";

import Link from 'next/link';
import type { BlogPost, BlogCategory, SocialLink, BlogTag } from '@/lib/data';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Mail, Rss, Tag as TagIcon, Layers, UsersRound, Search } from 'lucide-react'; 
import { personalInfo, getIcon } from '@/lib/data';
import { Skeleton } from '../ui/skeleton';
import { Alert, AlertDescription } from '../ui/alert';
import { AdSensePlaceholder } from '@/components/AdSensePlaceholder'; // Import reusable component

interface BlogSidebarProps {
  recentPosts: BlogPost[];
  categories: BlogCategory[];
  tags: BlogTag[];
  socialLinks: SocialLink[] | null;
  isLoadingSocialLinks?: boolean;
  socialLinksError?: string | null;
}

export function BlogSidebar({ 
  recentPosts, 
  categories, 
  tags,
  socialLinks,
  isLoadingSocialLinks,
  socialLinksError 
}: BlogSidebarProps) {

  const SocialLinksContent = () => {
    if (isLoadingSocialLinks) {
      return (
        <div className="flex flex-wrap justify-center gap-3">
          {[...Array(4)].map((_, i) => (
            <Skeleton key={i} className="h-9 w-9 rounded-full" />
          ))}
        </div>
      );
    }
    if (socialLinksError && (!socialLinks || socialLinks.length === 0)) { 
      return <AlertDescription className="text-xs text-destructive">Social links unavailable.</AlertDescription>;
    }
    if (!socialLinks || socialLinks.length === 0) {
      return <AlertDescription className="text-xs">No social links to display.</AlertDescription>;
    }
    return (
      <div className="flex flex-wrap justify-center gap-3">
        {socialLinks.map((link) => {
           if (!link || !link.iconName) {
              console.warn("BlogSidebar: Malformed social link or missing iconName:", link);
              return null;
            }
          const IconComponent = getIcon(link.iconName);
          return (
            <Link key={link.name} href={link.url} target="_blank" rel="noopener noreferrer" aria-label={`Visit ${personalInfo.name}'s ${link.name} profile`}>
              <Button variant="outline" size="icon" className="text-muted-foreground hover:text-primary rounded-full">
                {IconComponent && <IconComponent className="h-5 w-5" />}
              </Button>
            </Link>
          );
        })}
      </div>
    );
  };

  return (
    <div className="space-y-8">
      {/* Search Bar Placeholder */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Search</CardTitle>
        </CardHeader>
        <CardContent>
          <form className="flex gap-2">
            <Input type="search" placeholder="Search posts..." className="flex-grow" />
            <Button type="submit" variant="outline" size="icon" aria-label="Search">
              <Search className="h-4 w-4" /> 
            </Button>
          </form>
        </CardContent>
      </Card>

      <AdSensePlaceholder slot="Blog Sidebar - Top" />

      {/* Recent Posts */}
      {recentPosts && recentPosts.length > 0 ? (
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Recent Posts</CardTitle>
          </CardHeader>
          <CardContent>
            <ul className="space-y-3">
              {recentPosts.map(post => (
                <li key={post.id}> {/* Changed key to post.id */}
                  <Link href={`/blog/${post.slug}`} className="text-sm text-primary hover:underline">
                    {post.frontmatter.title}
                  </Link>
                  <p className="text-xs text-muted-foreground">
                    {new Date(post.frontmatter.date).toLocaleDateString('en-US', { month: 'short', day: 'numeric' })}
                  </p>
                </li>
              ))}
            </ul>
          </CardContent>
        </Card>
      ) : (
        <Card>
          <CardHeader><CardTitle className="text-lg">Recent Posts</CardTitle></CardHeader>
          <CardContent><p className="text-sm text-muted-foreground">No recent posts.</p></CardContent>
        </Card>
      )}

      {/* Categories */}
      {categories && categories.length > 0 ? (
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Categories</CardTitle>
          </CardHeader>
          <CardContent>
            <ul className="space-y-2">
              {categories.map(category => (
                <li key={category.slug} className="flex justify-between items-center">
                  <Link href={`/blog/category/${category.slug}`} className="text-sm text-primary hover:underline">
                    {category.name}
                  </Link>
                  <span className="text-xs bg-muted px-1.5 py-0.5 rounded-sm text-muted-foreground">{category.count}</span>
                </li>
              ))}
            </ul>
          </CardContent>
        </Card>
      ) : (
         <Card>
          <CardHeader><CardTitle className="text-lg">Categories</CardTitle></CardHeader>
          <CardContent><p className="text-sm text-muted-foreground">No categories found.</p></CardContent>
        </Card>
      )}
      
      {/* Tags */}
      {tags && tags.length > 0 ? (
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Tags</CardTitle>
          </CardHeader>
          <CardContent className="flex flex-wrap gap-2">
            {tags.map(tag => (
              <Link key={tag.slug} href={`/blog/tag/${tag.slug}`}>
                <Button variant="outline" size="sm" className="text-xs hover:bg-accent/10 hover:text-accent hover:border-accent">
                  <TagIcon className="mr-1 h-3 w-3" />
                  {tag.name} {tag.count > 0 ? `(${tag.count})` : ''}
                </Button>
              </Link>
            ))}
          </CardContent>
        </Card>
      ) : (
        <Card>
          <CardHeader><CardTitle className="text-lg">Tags</CardTitle></CardHeader>
          <CardContent><p className="text-sm text-muted-foreground">No tags found.</p></CardContent>
        </Card>
      )}

      <AdSensePlaceholder slot="Blog Sidebar - Middle" />

      {/* Newsletter Signup Placeholder */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Newsletter</CardTitle>
        </CardHeader>
        <CardContent>
          <p className="text-sm text-muted-foreground mb-3">Stay updated with my latest articles and insights.</p>
          <form className="flex flex-col gap-2">
            <Input type="email" placeholder="your.email@example.com" />
            <Button type="submit">Subscribe</Button>
          </form>
        </CardContent>
      </Card>

      {/* Social Links */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Connect</CardTitle>
        </CardHeader>
        <CardContent>
          <SocialLinksContent />
        </CardContent>
      </Card>

       <Card>
          <CardHeader>
            <CardTitle className="text-lg">About Me</CardTitle>
          </CardHeader>
          <CardContent className="space-y-2">
             <Link href="/#about" className="flex items-center text-primary hover:underline">
                <UsersRound className="mr-2 h-4 w-4"/> My Resume
             </Link>
             <Link href="/#contact" className="flex items-center text-primary hover:underline">
                <Mail className="mr-2 h-4 w-4"/> Hire Me
             </Link>
          </CardContent>
        </Card>
        <AdSensePlaceholder slot="Blog Sidebar - Bottom" />
    </div>
  );
}
