
"use client";

import { useState, useEffect, ReactNode } from "react";
import { ThemeProvider } from "@/components/ThemeProvider";
import { Toaster } from "@/components/ui/toaster";
import { GoogleAnalytics } from "@/components/GoogleAnalytics";
import { InitialLoadingScreen } from "@/components/layout/InitialLoadingScreen";
import { SplashScreen } from "@/components/layout/SplashScreen";
import { CookieConsentBanner } from "@/components/layout/CookieConsentBanner";

interface ClientLayoutWrapperProps {
  children: ReactNode;
  gaTrackingId?: string;
}

export function ClientLayoutWrapper({ children, gaTrackingId }: ClientLayoutWrapperProps) {
  const [isTopLevel, setIsTopLevel] = useState(false);
  const [isLoadingInitial, setIsLoadingInitial] = useState(true);
  const [showSplashScreen, setShowSplashScreen] = useState(false);

  useEffect(() => {
    // This check prevents the loading/splash screens from running inside iframes
    const topLevel = window.self === window.top;
    setIsTopLevel(topLevel);

    if (topLevel) {
      // Logic for top-level window
      document.documentElement.style.overflow = 'hidden'; // Disable scroll during initial load

      const timer = setTimeout(() => {
        setIsLoadingInitial(false);
        setShowSplashScreen(true);
      }, 2800); // Duration of the duck loading screen

      return () => clearTimeout(timer);
    } else {
      // Logic for iframes
      setIsLoadingInitial(false);
      setShowSplashScreen(false);
      document.documentElement.style.overflow = 'auto'; // Ensure iframe content can scroll
    }
  }, []);

  const handleSplashChoiceMade = () => {
    setShowSplashScreen(false);
    document.documentElement.style.overflow = 'auto'; // Re-enable scroll
  };
  
  // This determines if any overlay (loading or splash) should be active.
  const showOverlays = isTopLevel && (isLoadingInitial || showSplashScreen);

  return (
    <ThemeProvider>
      {gaTrackingId && <GoogleAnalytics gaTrackingId={gaTrackingId} />}
      
      {/* 
        This div wraps the main content. We use visibility instead of conditional rendering 
        to prevent layout shifts and ensure the page structure is always present for SEO, 
        even when overlays are active.
      */}
      <div style={{ visibility: showOverlays ? 'hidden' : 'visible' }}>
        {children}
      </div>

      {/* Overlays that are managed by this component's state */}
      {isLoadingInitial && isTopLevel && <InitialLoadingScreen />}
      {showSplashScreen && isTopLevel && <SplashScreen onChoiceMade={handleSplashChoiceMade} />}
      
      {/* Independent components that manage their own state, rendered on top */}
      <Toaster />
      {isTopLevel && <CookieConsentBanner />}
    </ThemeProvider>
  );
}
