
import type { BlogPost } from "@/lib/data";
import { SectionWrapper } from "@/components/SectionWrapper";
import { BlogPostCard } from "@/components/blog/BlogPostCard";
import { Button } from "@/components/ui/button";
import Link from "next/link";
import { ArrowRight, AlertCircle } from "lucide-react";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { Skeleton } from "@/components/ui/skeleton";

interface LatestBlogPostsSectionProps {
  posts: BlogPost[] | null;
  isLoading?: boolean;
  error?: string | null;
}

export function LatestBlogPostsSection({ posts, isLoading, error }: LatestBlogPostsSectionProps) {
  if (isLoading) {
    return (
      <SectionWrapper id="latest-blog" title="Latest Articles" subtitle="Catch up on my newest thoughts and insights.">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {[...Array(3)].map((_, i) => (
            <div key={i} className="flex flex-col space-y-3 p-4 border rounded-lg bg-card/50">
              <Skeleton className="h-[180px] w-full rounded-md" />
              <div className="space-y-2 pt-2">
                <Skeleton className="h-5 w-3/4 rounded" />
                <Skeleton className="h-4 w-1/2 rounded mt-1" />
                <Skeleton className="h-3 w-full rounded mt-2" />
                <Skeleton className="h-3 w-5/6 rounded mt-1" />
              </div>
            </div>
          ))}
        </div>
      </SectionWrapper>
    );
  }

  if (error) {
    return (
      <SectionWrapper id="latest-blog" title="Latest Articles">
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Recent Posts</AlertTitle>
          <AlertDescription>
            {error} Please try again later.
          </AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  if (!posts || posts.length === 0) {
    return (
      <SectionWrapper id="latest-blog" title="Latest Articles" subtitle="Catch up on my newest thoughts and insights.">
        <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Recent Articles</AlertTitle>
          <AlertDescription>
            There are currently no recent blog posts to display. Check back soon!
          </AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  return (
    <SectionWrapper id="latest-blog" title="Latest Articles" subtitle="Catch up on my newest thoughts and insights.">
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
        {posts.map((post) => (
          <BlogPostCard key={post.slug} post={post} />
        ))}
      </div>
      <div className="mt-12 text-center">
        <Link href="/blog">
          <Button variant="outline" size="lg">
            View All Posts <ArrowRight className="ml-2 h-4 w-4" />
          </Button>
        </Link>
      </div>
    </SectionWrapper>
  );
}
