
import Link from "next/link";
import { SectionWrapper } from "@/components/SectionWrapper";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { personalInfo, type PatreonTier } from "@/lib/data";
import { Heart, CheckCircle, AlertCircle } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";

interface PatreonSupportSectionProps {
  patreonTiers: PatreonTier[] | null;
  isLoading: boolean;
  error: string | null;
}

export function PatreonSupportSection({ patreonTiers, isLoading, error }: PatreonSupportSectionProps) {
  if (isLoading) {
    return (
      <SectionWrapper id="patreon" title="Support My Work" subtitle="Join my developer community...">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
          {[...Array(3)].map((_, i) => (
            <Card key={i} className="shadow-lg">
              <CardHeader className="p-4">
                <div className="flex items-center gap-3">
                  <Skeleton className="h-7 w-7 rounded" />
                  <Skeleton className="h-6 w-3/4" />
                </div>
                <Skeleton className="h-7 w-1/3 mt-1" />
              </CardHeader>
              <CardContent className="p-4 pt-2">
                <div className="flex items-start space-x-2">
                  <Skeleton className="h-5 w-5 rounded-full mt-1" />
                  <Skeleton className="h-4 w-full" />
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
        <div className="text-center">
          <Skeleton className="h-12 w-48 rounded-full mx-auto" />
          <Skeleton className="h-3 w-3/4 mt-4 mx-auto" />
        </div>
      </SectionWrapper>
    );
  }
  
  if (error) {
    return (
      <SectionWrapper id="patreon" title="Support My Work" subtitle="Join my developer community...">
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Support Tiers</AlertTitle>
          <AlertDescription>{error}</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  if (!patreonTiers || patreonTiers.length === 0) {
    return (
      <SectionWrapper id="patreon" title="Support My Work" subtitle="Join my developer community...">
         <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Support Tiers Available</AlertTitle>
          <AlertDescription>Support tier information is currently not available.</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }
  
  return (
    <SectionWrapper 
      id="patreon" 
      title="Support My Work" 
      subtitle="Join my developer community on Patreon to get early access to tutorials, behind-the-scenes content, and direct support. Your contribution helps me create more valuable content for everyone!"
    >
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
        {patreonTiers.map((tier: PatreonTier) => {
          const IconComponent = tier.icon || Heart;
          return (
            <Card 
              key={tier.name} 
              className="shadow-xl hover:shadow-2xl transition-shadow duration-300 flex flex-col bg-card text-card-foreground"
            >
              <CardHeader className="bg-muted/30 dark:bg-muted/20 p-4">
                <div className="flex items-center gap-3">
                   <span className="bg-clip-text text-transparent bg-gradient-to-r from-[hsl(var(--primary))] via-[hsl(var(--shimmer-highlight))] to-[hsl(var(--accent))] bg-[size:300%_300%] animate-gradient-xy inline-flex items-center justify-center">
                    <IconComponent className="h-7 w-7" aria-hidden="true" />
                  </span>
                  <CardTitle className="text-xl font-semibold text-primary">{tier.name}</CardTitle>
                </div>
                <CardDescription className="text-2xl font-bold text-card-foreground pt-1">{tier.price}</CardDescription>
              </CardHeader>
              <CardContent className="flex-grow p-4 pt-2">
                <div className="flex items-start space-x-2">
                  <span className="bg-clip-text text-transparent bg-gradient-to-r from-green-500 via-[hsl(var(--shimmer-highlight))] to-green-300 bg-[size:300%_300%] animate-gradient-xy inline-flex items-center justify-center mt-1 flex-shrink-0">
                    <CheckCircle className="h-5 w-5" aria-hidden="true" />
                  </span>
                  <p className="text-sm text-muted-foreground">{tier.benefits}</p>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>
      <div className="text-center">
        <Link href={personalInfo.email ? `mailto:${personalInfo.email}?subject=Support%20Inquiry` : "#"} target="_blank" rel="noopener noreferrer">
          <Button size="lg" className="bg-primary hover:bg-primary/90 text-primary-foreground rounded-full px-8 py-3 text-lg">
            <Heart className="mr-2 h-5 w-5" /> Support My Work
          </Button>
        </Link>
        <p className="mt-4 text-sm text-foreground/80">
          Every bit of support is greatly appreciated and helps me dedicate more time to creating helpful resources.
        </p>
      </div>
    </SectionWrapper>
  );
}
