
"use client";

import { useState } from "react";
import { SectionWrapper } from "@/components/SectionWrapper";
import type { Project } from "@/lib/data";
import { ProjectDetailModal } from "@/components/ProjectDetailModal";
import { ProjectCard } from "@/components/ProjectCard";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { AlertCircle } from "lucide-react";

interface ProjectsSectionProps {
  projects: Project[] | null;
  isLoading: boolean;
  error: string | null;
}

export function ProjectsSection({ projects, isLoading, error }: ProjectsSectionProps) {
  const [selectedProject, setSelectedProject] = useState<Project | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);

  const openModal = (project: Project) => {
    console.log("ProjectsSection: Opening modal for project -", project?.title);
    setSelectedProject(project);
    setIsModalOpen(true);
  };

  const closeModal = () => {
    console.log("ProjectsSection: Closing modal.");
    setSelectedProject(null);
    setIsModalOpen(false);
  };

  if (isLoading) {
    return (
      <SectionWrapper
        id="projects"
        title="Featured Projects"
        subtitle="A selection of web projects I have delivered."
      >
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {[...Array(3)].map((_, i) => (
            <div key={i} className="flex flex-col space-y-3 p-4 border rounded-lg bg-card/50">
              <Skeleton className="h-[180px] w-full rounded-xl" />
              <div className="space-y-2 pt-2">
                <Skeleton className="h-4 w-3/4 rounded" />
                <Skeleton className="h-4 w-1/2 rounded mt-1" />
              </div>
              <div className="flex flex-wrap gap-2 mt-2">
                <Skeleton className="h-6 w-20 rounded-full" />
                <Skeleton className="h-6 w-24 rounded-full" />
              </div>
              <Skeleton className="h-9 w-28 mt-4" />
            </div>
          ))}
        </div>
      </SectionWrapper>
    );
  }

  if (error) { 
    return (
      <SectionWrapper
        id="projects"
        title="Featured Projects"
        subtitle="A selection of web projects I have delivered."
      >
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Projects</AlertTitle>
          <AlertDescription>
            {error} Please try again later.
          </AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }
  
  if (!projects || projects.length === 0) {
    return (
      <SectionWrapper
        id="projects"
        title="Featured Projects"
        subtitle="A selection of web projects I have delivered."
      >
         <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Projects Available</AlertTitle>
          <AlertDescription>
            There are currently no projects to display. Check back later!
          </AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  return (
    <SectionWrapper
      id="projects"
      title="Featured Projects"
      subtitle="A selection of web projects I have delivered."
    >
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
        {projects.map((project, index) => {
          if (!project || typeof project.title === 'undefined') {
            console.error("ProjectsSection: Skipping malformed project data at index:", index, project);
            return (
              <div key={`error-prop-${index}`} className="p-4 border border-red-700 bg-red-200 text-red-800 rounded-lg h-full">
                Error: Malformed project data (from props) at index {index}. Title is missing.
              </div>
            );
          }
           console.log("ProjectsSection: Mapping project -", project.title, "at index", index);
          return (
            <ProjectCard
              key={`${project.id || project.title}-prop-${index}`}
              project={project}
              onViewDetails={() => openModal(project)}
            />
          );
        })}
      </div>
      {selectedProject && (
        <ProjectDetailModal
          project={selectedProject}
          isOpen={isModalOpen}
          onClose={closeModal}
        />
      )}
    </SectionWrapper>
  );
}
