
import Image from "next/image";
import { SectionWrapper } from "@/components/SectionWrapper";
import { Card, CardContent, CardFooter, CardHeader } from "@/components/ui/card";
import type { Testimonial } from "@/lib/data";
import { Star, Quote, AlertCircle } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";

interface TestimonialsSectionProps {
  testimonials: Testimonial[] | null;
  isLoading: boolean;
  error: string | null;
}

export function TestimonialsSection({ testimonials, isLoading, error }: TestimonialsSectionProps) {
  if (isLoading) {
    return (
      <SectionWrapper id="testimonials" title="What People Say" subtitle="Kind words from colleagues and clients I've worked with.">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
          {[...Array(2)].map((_, i) => (
            <Card key={i} className="shadow-lg">
              <CardHeader className="pb-2 pt-4 px-4">
                <div className="flex items-center mb-2">
                  {[...Array(5)].map((_, j) => ( <Skeleton key={j} className="h-5 w-5 mr-0.5 rounded-sm" /> ))}
                </div>
              </CardHeader>
              <CardContent className="p-4">
                <Skeleton className="h-6 w-6 mb-2" />
                <Skeleton className="h-3 w-full mb-1" />
                <Skeleton className="h-3 w-full mb-1" />
                <Skeleton className="h-3 w-3/4 mb-4" />
              </CardContent>
              <CardFooter className="mt-auto pt-4 p-4 border-t">
                <Skeleton className="h-12 w-12 rounded-full mr-4" />
                <div className="space-y-1">
                  <Skeleton className="h-4 w-32" />
                  <Skeleton className="h-3 w-24" />
                </div>
              </CardFooter>
            </Card>
          ))}
        </div>
      </SectionWrapper>
    );
  }

  if (error) {
    return (
      <SectionWrapper id="testimonials" title="What People Say" subtitle="Kind words from colleagues and clients I've worked with.">
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Testimonials</AlertTitle>
          <AlertDescription>{error}</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }
  
  if (!testimonials || testimonials.length === 0) {
    return (
      <SectionWrapper id="testimonials" title="What People Say" subtitle="Kind words from colleagues and clients I've worked with.">
         <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Testimonials Available</AlertTitle>
          <AlertDescription>Testimonial information is currently not available.</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  return (
    <SectionWrapper 
      id="testimonials" 
      title="What People Say" 
      subtitle="Kind words from colleagues and clients I've worked with."
    >
      <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
        {testimonials.map((testimonial: Testimonial, index: number) => (
          <Card 
            key={index} 
            className="shadow-xl hover:shadow-2xl transition-shadow duration-300 flex flex-col bg-card text-card-foreground"
          >
            <CardHeader className="pb-2 pt-4 px-4">
              <div className="flex items-center mb-2">
                {[...Array(5)].map((_, i) => (
                  <Star
                    key={i}
                    className={`h-5 w-5 ${i < testimonial.rating ? "text-yellow-400 fill-yellow-400" : "text-muted-foreground/50"}`}
                    aria-hidden="true"
                  />
                ))}
              </div>
            </CardHeader>
            <CardContent className="flex-grow p-4">
              <span className="bg-clip-text text-transparent bg-gradient-to-r from-[hsl(var(--primary))] via-[hsl(var(--shimmer-highlight))] to-[hsl(var(--accent))] bg-[size:300%_300%] animate-gradient-xy inline-flex items-center justify-center mb-2">
                <Quote className="h-8 w-8 transform -scale-x-100" aria-hidden="true" />
              </span>
              <blockquote className="text-card-foreground/90 italic mb-4">
                "{testimonial.quote}"
              </blockquote>
            </CardContent>
            <CardFooter className="mt-auto pt-4 p-4 border-t">
              <div className="flex items-center">
                <div className="relative w-12 h-12 rounded-full overflow-hidden mr-4">
                  <Image
                    src={testimonial.image}
                    alt={testimonial.author}
                    fill
                    style={{objectFit: "cover"}}
                    sizes="48px"
                    data-ai-hint={testimonial.dataAiHint}
                  />
                </div>
                <div>
                  <p className="font-semibold text-card-foreground">{testimonial.author}</p>
                  <p className="text-sm text-muted-foreground">{testimonial.title}</p>
                </div>
              </div>
            </CardFooter>
          </Card>
        ))}
      </div>
    </SectionWrapper>
  );
}
