
import fs from 'fs';
import path from 'path';
import matter from 'gray-matter';
import { remark } from 'remark';
import html from 'remark-html';
import type { BlogPost, BlogPostMatter } from './data';

const postsDirectory = path.join(process.cwd(), 'src/content/blog');

export async function getPostSlugs() {
  const filenames = fs.readdirSync(postsDirectory);
  return filenames.map(filename => filename.replace(/\.md$/, ''));
}

export async function getPostBySlug(slug: string): Promise<BlogPost | null> {
  const fullPath = path.join(postsDirectory, `${slug}.md`);
  try {
    const fileContents = fs.readFileSync(fullPath, 'utf8');
    const { data, content } = matter(fileContents);

    // Process markdown content to HTML
    const processedContent = await remark().use(html).process(content);
    const contentHtml = processedContent.toString();

    return {
      slug,
      frontmatter: data as BlogPostMatter,
      content: contentHtml,
    };
  } catch (error) {
    console.error(`Error reading or parsing post ${slug}:`, error);
    return null; // Or re-throw, or handle as appropriate
  }
}

export async function getAllPosts(): Promise<BlogPost[]> {
  const slugs = await getPostSlugs();
  const postsPromises = slugs.map(slug => getPostBySlug(slug));
  const posts = (await Promise.all(postsPromises)).filter(post => post !== null) as BlogPost[];

  // Sort posts by date in descending order
  return posts.sort((post1, post2) => (post1.frontmatter.date > post2.frontmatter.date ? -1 : 1));
}

export function getRecentPosts(allPosts: BlogPost[], count: number): BlogPost[] {
  return allPosts.slice(0, count);
}

export function getFeaturedPosts(allPosts: BlogPost[]): BlogPost[] {
  return allPosts.filter(post => post.frontmatter.featured === true);
}

export function getAllCategories(allPosts: BlogPost[]): { name: string; count: number }[] {
  const categoryCounts: Record<string, number> = {};
  allPosts.forEach(post => {
    const category = post.frontmatter.category;
    categoryCounts[category] = (categoryCounts[category] || 0) + 1;
  });
  return Object.entries(categoryCounts).map(([name, count]) => ({ name, count }))
    .sort((a, b) => b.count - a.count);
}

export function getAllTags(allPosts: BlogPost[]): { name: string; count: number }[] {
  const tagCounts: Record<string, number> = {};
  allPosts.forEach(post => {
    post.frontmatter.tags?.forEach(tag => {
      tagCounts[tag] = (tagCounts[tag] || 0) + 1;
    });
  });
  return Object.entries(tagCounts).map(([name, count]) => ({ name, count }))
    .sort((a, b) => b.count - a.count);
}

// Note: For this to work, you'll need to install gray-matter, remark, and remark-html:
// npm install gray-matter remark remark-html
// yarn add gray-matter remark remark-html

