
import type { LucideIcon } from 'lucide-react';
import {
  Github, Linkedin, Mail, Briefcase, GraduationCap, Users, Code, Settings, Cloud, Package, Server, Database, Shield, MonitorPlay, Palette, Brain, MessageSquare, Heart, Award, BookOpen, Star, ExternalLink, Download, Moon, Sun, Menu, X, Phone as PhoneIcon, Link as LinkIcon, Link2,
  Workflow, Layers, ShieldAlert, UsersRound, Handshake, MessagesSquare, Code2, Network, PiggyBank, Target, Megaphone, ClipboardCheck, GitMerge, FileText, Globe, ListChecks, Repeat, Presentation, DollarSign, PenToolIcon, TerminalSquare, ListTodo, IterationCw,
  Coffee, Gamepad2, Film, Music, Plane, Smile, Rss, LayoutList, Tags as TagsIcon, Building, Anchor, Atom, Activity, Home, BookUser, ContactIcon, Newspaper, ThumbsUp, Send, LogIn, UserPlus, Chrome, Facebook, Eye
} from 'lucide-react';

// --- Generic API Response ---
export interface ApiResponse<T> {
  success: boolean;
  data: T | null;
  error?: string | null;
  statusCode?: number | null; // Can be null if network error occurs before response
}

// --- Personal Info (Remains Static for now due to widespread metadata use & fallback) ---
export const personalInfo = {
  name: "Hussain Abuhajajj",
  title: "Software Project Manager | Former Full Stack Web Developer",
  phone: "+47 486 79 240",
  email: "Hussain.h.ff32@gmail.com",
  about: "Experienced Software Project Manager with a strong background in full-stack web development and entrepreneurial leadership. Proven ability to lead cross-functional teams, manage full project lifecycles, and deliver scalable, secure, and user-focused solutions. Adept at aligning business goals with technology strategies, ensuring timely and quality delivery of software projects.",
  social: {
    linkedin: "https://www.linkedin.com/in/hussainabuhajajj/",
    github: "https://github.com/hussainabuhajajj",
    twitter: "https://twitter.com/hussainabuhajaj",
  },
  profileImage: "https://placehold.co/300x300.png",
  siteUrl: "https://hussainwh.info",
  avatarImage: "https://assets.zyrosite.com/cdn-cgi/image/format=auto,w=656,h=656,fit=crop/mP4eW5e5e0T0PkWW/chatrp-png-YKbLDRx50xHBEj00.png",
  defaultOgImage: "https://hussainwh.info/og-image.png",
};

// --- App Settings ---
export interface AppSettingsApiItem {
  id: string | number;
  title: string;
  description: string;
  organization_name?: string;
  logo?: string;
  logo_url?: string;
  favicon?: string;
  favicon_url?: string;
  google_console_code?: string;
  google_analytic_code?: string;
  google_adsense_code?: string;
  google_tag_manager_code?: string;
  quick_links?: string;
  created_at?: string;
  updated_at?: string;
}
export interface AppSettingsApiResponse {
    data: AppSettingsApiItem[];
}
export interface AppSettings extends AppSettingsApiItem {}


// --- API Resource Interfaces & Frontend Data Types ---

// Navigation Links
export interface NavigationLinkApiItem {
  id: number | string;
  name: string;
  url: string;
  icon_name: string | null;
  order?: number;
}
export interface NavigationLinksApiResponse {
  data: NavigationLinkApiItem[];
}
export interface NavigationLink {
  name: string;
  href: string;
  iconName: keyof typeof iconMap | string;
}

export const defaultNavigationLinks: NavigationLink[] = [
  { name: "Home", href: "/#hero", iconName: "Home" },
  { name: "About", href: "/#about", iconName: "BookUser" },
  { name: "Skills", href: "/#stack", iconName: "Layers" },
  { name: "Experience", href: "/#experience", iconName: "Briefcase" },
  { name: "Education", href: "/#education", iconName: "GraduationCap" },
  { name: "Projects", href: "/#projects", iconName: "Code2" },
  { name: "Blog", href: "/blog", iconName: "Newspaper" },
  { name: "Contact", href: "/#contact", iconName: "ContactIcon" },
];


// Social Links
export interface SocialLinkApiItem {
  id: number | string;
  name: string;
  url: string;
  icon_name: string | null;
  order?: number;
}
export interface SocialLinksApiResponse {
  data: SocialLinkApiItem[];
}
export interface SocialLink {
  name:string;
  url: string;
  iconName: keyof typeof iconMap | string;
}
export const defaultSocialLinks: SocialLink[] = [
  { name: "LinkedIn", url: personalInfo.social.linkedin, iconName: "Linkedin" },
  { name: "GitHub", url: personalInfo.social.github, iconName: "Github" },
  { name: "Email", url: `mailto:${personalInfo.email}`, iconName: "Mail" },
  { name: "Twitter", url: personalInfo.social.twitter, iconName: "Twitter" },
];


// Tech Stack
export interface TechItemApiResource {
  id: number | string;
  name: string;
  icon_url_or_svg: string | null; // Expected to be an icon name string
  proficiency_level?: number;
  order?: number;
  category?: any; // Not typically used on frontend for TechStackItem
}

export interface TechCategoryApiItem {
  id: number | string;
  name: string;
  order?: number;
  tech_items: TechItemApiResource[];
}

export interface TechStackApiResponse {
  data: TechCategoryApiItem[];
}

export interface TechStackItem {
  name: string;
  icon: LucideIcon;
}
export interface TechStackCategory {
  category: string;
  items: TechStackItem[];
}


// Experience
export interface ExperienceDescriptionPointApiItem {
    id: number | string;
    point_text: string;
    order?: number;
}
export interface ExperienceApiItem {
  id: number | string;
  title: string; // Role
  company_name: string;
  company_website?: string | null;
  location?: string | null;
  start_date: string | null;
  end_date: string | null;
  is_current?: boolean;
  short_description?: string | null;
  order?: number;
  description_points: ExperienceDescriptionPointApiItem[];
  created_at?: string;
  updated_at?: string;
}
export interface ExperienceApiResponse {
  data: ExperienceApiItem[];
}
export interface ExperienceEntry {
  role: string;
  company: string;
  dates: string;
  description: string[];
  icon: LucideIcon;
  company_website?: string;
  location?: string;
  short_description?: string;
}


// Education
export interface EducationApiItem {
  id: number | string;
  institution_name: string;
  degree_or_certificate: string;
  field_of_study: string | null;
  start_date: string | null;
  end_date: string | null;
  grade_or_score: string | null;
  description: string | null;
  order?: number;
  created_at?: string;
  updated_at?: string;
  // icon_name is no longer expected from API for education items as per last update
}
export interface EducationApiResponse {
    data: EducationApiItem[];
}
export interface EducationEntry {
  id: number | string;
  item: string;
  institution: string;
  dates: string;
  icon: LucideIcon;
  type: 'education' | 'certification';
  description?: string;
  fieldOfStudy?: string;
  gradeOrScore?: string;
}


// Projects
export interface TechnologyResource {
  id: number;
  name: string;
  slug: string;
  icon_name: string | null;
  icon_svg?: string | null;
}
export interface ProjectApiItem {
  id: number;
  name: string;
  description: string;
  image_url: string | null;
  project_url: string | null;
  repo_url: string | null;
  start_date: string | null;
  end_date: string | null;
  is_featured?: boolean;
  order?: number;
  created_at?: string;
  updated_at?: string;
  technologies?: TechnologyResource[];
  photo_alt_text?: string | null;
}
export interface ProjectsApiResponse {
    data: ProjectApiItem[];
}
export interface ProjectTech {
  name: string;
  icon: LucideIcon;
}
export interface Project {
  id: number;
  title: string;
  description: string;
  url?: string;
  image: string;
  dataAiHint: string;
  tech: ProjectTech[];
}


// Interests
export interface InterestApiItem {
  id: number | string;
  name: string;
  icon_name: string | null;
  description?: string;
  order?: number;
}
export interface InterestsApiResponse {
  data: InterestApiItem[];
}
export interface Interest {
  name: string;
  icon: LucideIcon;
  description?: string;
}

// Testimonials
export interface TestimonialApiItem {
  id: number | string;
  quote: string;
  author: string;
  title: string;
  image_url: string | null;
  rating: number;
  data_ai_hint?: string;
  order?: number;
}
export interface TestimonialsApiResponse {
  data: TestimonialApiItem[];
}
export interface Testimonial {
  quote: string;
  author: string;
  title: string;
  image: string;
  rating: number;
  dataAiHint: string;
}

// Patreon Tiers
export interface PatreonTierApiItem {
  id: number | string;
  name: string;
  price: string;
  benefits: string;
  icon_name: string | null;
  order?: number;
  link_url?: string;
}
export interface PatreonTiersApiResponse {
  data: PatreonTierApiItem[];
}
export interface PatreonTier {
  name: string;
  price: string;
  benefits: string;
  icon?: LucideIcon;
  linkUrl?: string;
}


// Contact Form Submission
export interface ContactFormPayload {
  name: string;
  email: string;
  subject?: string | null;
  message: string;
}
export interface ContactSubmissionResource {
    id: number;
    name: string;
    email: string;
    subject: string | null;
    message: string;
    is_read: boolean;
    ip_address: string | null;
    user_agent: string | null;
    submitted_at: string; // ISO 8601 date string
    created_at: string | null; // ISO 8601 date string
    updated_at: string | null; // ISO 8601 date string
}


// --- Blog Related API Interfaces ---
export interface UserResource {
  id: number | string;
  name: string;
  email: string;
  email_verified_at?: string | null;
  created_at?: string;
  updated_at?: string;
  profile_photo_path?: string | null;
  avatar_url?: string | null;
}

export interface TagResource {
  id: string | number;
  name: string;
  slug: string;
  posts_count?: number;
  links?: { self: string };
}
export interface TagsApiResponse {
    data: TagResource[];
    links?: any;
    meta?: any;
}


export interface CategoryApiItem {
  id: string | number;
  name: string;
  slug: string;
  posts_count?: number;
  posts?: Partial<PostResource>[]; // Assuming posts might not be fully populated here
  links?: { self: string };
}
export interface CategoriesApiResponse {
    data: CategoryApiItem[];
    links?: { first?: string | null; last?: string | null; prev?: string | null; next?: string | null; };
    meta?: { current_page?: number; from?: number | null; last_page?: number; links?: Array<{ url: string | null; label: string; active: boolean }>; path?: string; per_page?: number; to?: number | null; total?: number; };
}

export interface SeoDetailResource {
  id: string | number;
  post_id?: number | string;
  post?: Partial<PostResource>; // Avoid circular full dependency
  title: string;
  keywords: string[] | string; // Can be array or comma-separated string
  description: string;
  created_at?: string;
  updated_at?: string;
  links?: { self: string };
}

export type CommentSentiment = "positive" | "negative" | "neutral";

export interface CommentResource {
  id: string | number;
  post_id: string | number;
  user?: UserResource;
  name?: string; // For anonymous comments
  email?: string; // For anonymous comments
  comment: string;
  approved: boolean;
  approved_at: string | null;
  created_at: string;
  updated_at: string;
  links?: { self: string };
  likes?: number; // Added for comment likes
  sentiment?: CommentSentiment; // Added for sentiment analysis
}

export interface PostResource {
  id: number | string;
  title: string;
  slug: string;
  sub_title: string | null;
  body: string;
  status: string;
  published_at: string; // Could be various formats, e.g., "DD MMM YYYY" or ISO
  scheduled_for?: string | null;
  is_featured?: boolean;
  show_top_ad?: boolean;
  show_bottom_ad?: boolean;
  affiliate_products?: Array<{
    name: string;
    review: string;
    link: string;
    image: string;
    dataAiHint: string;
    rating: number;
  }>;
  cover_photo_url: string | null;
  photo_alt_text: string | null;
  user: UserResource;
  categories: CategoryApiItem[];
  tags: TagResource[];
  comments_count?: number;
  comments?: CommentResource[];
  seo_detail?: SeoDetailResource | null;
  related_posts?: Partial<PostResource>[];
  links?: { self: string };
  likes?: number;
  views_count?: number;
}

export interface SinglePostApiResponse {
  data: PostResource;
}

export interface PostsApiResponse {
  data: PostResource[];
  links?: { first?: string | null; last?: string | null; prev?: string | null; next?: string | null; };
  meta?: { current_page?: number; from?: number | null; last_page?: number; links?: Array<{ url: string | null; label: string; active: boolean }>; path?: string; per_page?: number; to?: number | null; total?: number; };
}


// Frontend structure for a blog post
export interface BlogPostMatter {
  id: number | string;
  title: string;
  date: string; // This will store the string from published_at
  category: string;
  excerpt: string;
  author: string;
  image?: string;
  tags?: string[];
  dataAiHint?: string;
  featured?: boolean;
  showTopAd?: boolean;
  showBottomAd?: boolean;
  affiliateProducts?: Array<{
    name: string;
    review: string;
    link: string;
    image: string;
    dataAiHint: string;
    rating: number;
  }>;
  seo_detail?: SeoDetailResource | null;
  likes?: number;
  comments_count?: number;
  views_count?: number;
}

export interface BlogPost {
  id: number | string;
  slug: string;
  frontmatter: BlogPostMatter;
  content: string; // HTML content
  comments?: CommentResource[];
}

// Frontend structure for a blog category displayed in sidebar/lists
export interface BlogCategory {
  id: string | number;
  name: string;
  slug: string;
  count: number;
}

// Frontend structure for a blog tag displayed in sidebar/lists
export interface BlogTag {
  id: string | number;
  name: string;
  slug: string;
  count: number;
}

// Payload for submitting a new comment
export interface NewCommentPayload {
  name: string;
  email: string;
  comment: string;
}

// Expected API response when submitting a comment
export interface SubmitCommentApiResponse {
  data: CommentResource; // Assuming API returns the created comment
  message?: string;
}

// --- Authentication ---
export interface LoginPayload {
  email: string;
  password: string;
}

export interface RegisterPayload {
  name: string;
  email: string;
  password: string;
  // password_confirmation: string; // Often required by backend
}

export interface AuthUser { // Example structure for a logged-in user
  id: number | string;
  name: string;
  email: string;
  // Add other relevant user fields, e.g., roles, permissions
}

export interface AuthResponse { // Example API response for login/register
  user: AuthUser;
  token?: string; // If using token-based auth
  message?: string;
}

// --- Genkit Related Types ---

// For SuggestRelatedPostsFlow
export interface SimplePostInfo {
  title: string;
  slug: string;
  tags: string[];
  category: string;
  excerpt: string;
}

export interface SuggestRelatedPostsInput {
  currentPostTitle: string;
  currentPostTags: string[];
  currentPostCategory: string;
  currentPostExcerpt: string;
  allPosts: SimplePostInfo[];
}

export interface SuggestedPost {
  title: string;
  slug: string;
}
export interface SuggestRelatedPostsOutput {
  relatedPosts: SuggestedPost[];
}

// For AnalyzeCommentSentimentFlow
export interface AnalyzeCommentSentimentInput {
  commentText: string;
}

export interface AnalyzeCommentSentimentOutput {
  sentiment: CommentSentiment;
  confidence?: number;
}


// --- Icon Map (Utility, not API fetched) ---
export const iconMap: Record<string, LucideIcon> = {
  Github, Linkedin, Mail, Briefcase, GraduationCap, Users, Code, Settings, Cloud, Package, Server, Database, Shield, MonitorPlay, Palette, Brain, MessageSquare, Heart, Award, BookOpen, Star, ExternalLink, Download, Moon, Sun, Menu, X, PhoneIcon, LinkIcon, Link2,
  Workflow, Layers, ShieldAlert, UsersRound, Handshake, MessagesSquare, Code2, Network, PiggyBank, Target, Megaphone, ClipboardCheck, GitMerge, FileText, Globe, ListChecks, Repeat, Presentation, DollarSign, PenToolIcon, TerminalSquare, ListTodo, IterationCw,
  Coffee, Gamepad2, Film, Music, Plane, Smile, Rss, LayoutList, TagsIcon, Building, Anchor, Atom, Activity, Home, BookUser, ContactIcon, Newspaper, ThumbsUp, Send, LogIn, UserPlus, Chrome, Facebook, Eye,
  'github': Github, 'linkedin': Linkedin, 'mail': Mail, 'briefcase': Briefcase, 'graduation-cap': GraduationCap, 'graduationcap': GraduationCap, 'users': Users, 'code': Code, 'settings': Settings, 'cloud': Cloud, 'package': Package, 'server': Server, 'database': Database, 'shield': Shield, 'monitor-play': MonitorPlay, 'monitorplay': MonitorPlay, 'palette': Palette, 'brain': Brain, 'message-square': MessageSquare, 'messagesquare': MessageSquare, 'heart': Heart, 'award': Award, 'book-open': BookOpen, 'bookopen': BookOpen, 'star': Star, 'external-link': ExternalLink, 'externallink': ExternalLink, 'download': Download, 'moon': Moon, 'sun': Sun, 'menu': Menu, 'x': X, 'phone': PhoneIcon, 'phoneicon': PhoneIcon, 'link': LinkIcon, 'linkicon': LinkIcon, 'link2': Link2,
  'workflow': Workflow, 'layers': Layers, 'shield-alert': ShieldAlert, 'shieldalert': ShieldAlert, 'users-round': UsersRound, 'usersround': UsersRound, 'handshake': Handshake, 'messages-square': MessagesSquare, 'messagessquare': MessagesSquare, 'code-2': Code2, 'code2': Code2, 'network': Network, 'piggy-bank': PiggyBank, 'piggybank': PiggyBank, 'target': Target, 'megaphone': Megaphone, 'clipboard-check': ClipboardCheck, 'clipboardcheck': ClipboardCheck, 'git-merge': GitMerge, 'gitmerge': GitMerge, 'file-text': FileText, 'filetext': FileText, 'globe': Globe, 'list-checks': ListChecks, 'listchecks': ListChecks, 'repeat': Repeat, 'presentation': Presentation, 'dollar-sign': DollarSign, 'dollarsign': DollarSign, 'pen-tool': PenToolIcon, 'pentool': PenToolIcon, 'pentoolicon': PenToolIcon, 'terminal-square': TerminalSquare, 'terminalsquare': TerminalSquare, 'list-todo': ListTodo, 'listtodo': ListTodo, 'iteration-cw': IterationCw, 'iterationcw': IterationCw,
  'coffee': Coffee, 'gamepad-2': Gamepad2, 'gamepad2': Gamepad2, 'film': Film, 'music': Music, 'plane': Plane, 'smile': Smile, 'rss': Rss, 'layout-list': LayoutList, 'layoutlist': LayoutList, 'tags': TagsIcon, 'tagsicon': TagsIcon, 'building': Building, 'anchor': Anchor, 'atom': Atom, 'activity': Activity, 'home': Home, 'book-user': BookUser, 'bookuser': BookUser, 'contact': ContactIcon, 'contacticon': ContactIcon, 'newspaper': Newspaper, 'thumbsup': ThumbsUp, 'send': Send,
  'login': LogIn, 'userplus': UserPlus, 'chrome': Chrome, 'facebook': Facebook, 'eye': Eye,
  'defaulttech': Code,
  'default': Code,
};

// Helper function to get an icon by name, with a fallback
export function getIcon(iconName?: keyof typeof iconMap | string | null): LucideIcon {
  if (!iconName) {
    // console.warn("getIcon: No iconName provided. Using default 'Code' icon.");
    return Code;
  }
  // Normalize the icon name: convert to lowercase and remove non-alphanumeric characters
  // This helps match 'Code-2' or 'Code 2' with 'code2' in the map.
  const normalizedName = iconName.toLowerCase().replace(/[^a-z0-9]/gi, '');

  if (iconMap[normalizedName]) {
    return iconMap[normalizedName];
  }
  // Try original name if sanitized lookup fails (e.g. if map has 'MyIcon' and name is 'MyIcon')
  if (iconMap[iconName]) {
    return iconMap[iconName];
  }
  // console.warn(`getIcon: Icon not found for name: "${iconName}" (normalized: "${normalizedName}"). Using default 'Code' icon.`);
  return Code;
}

