
import axios, { type AxiosRequestConfig, type AxiosError, type InternalAxiosRequestConfig, type AxiosResponse } from 'axios';
import { API_BASE_URL, NEXT_APP_BASE_URL, PROXY_PATH } from '@/config';
import type {
  ApiResponse,
  AppSettings, AppSettingsApiItem, AppSettingsApiResponse,
  EducationEntry, EducationApiItem, EducationApiResponse,
  Project, ProjectApiItem, ProjectsApiResponse,
  BlogPost, PostResource, PostsApiResponse, SinglePostApiResponse,
  BlogCategory, CategoryApiItem, CategoriesApiResponse,
  BlogTag, TagResource, TagsApiResponse,
  ContactFormPayload, ContactSubmissionResource,
  TechStackCategory, TechCategoryApiItem, TechStackApiResponse,
  ExperienceEntry, ExperienceApiItem, ExperienceApiResponse,
  Testimonial, TestimonialApiItem, TestimonialsApiResponse,
  PatreonTier, PatreonTierApiItem, PatreonTiersApiResponse,
  Interest, InterestApiItem, InterestsApiResponse,
  NavigationLink, NavigationLinkApiItem, NavigationLinksApiResponse,
  SocialLink, SocialLinkApiItem, SocialLinksApiResponse,
  NewCommentPayload, SubmitCommentApiResponse, CommentResource,
  LoginPayload, RegisterPayload, AuthResponse, AuthUser
} from '@/lib/data';
import { getIcon, personalInfo } from '@/lib/data'; // Assuming personalInfo might be used for some fallbacks if needed
import { GraduationCap, Briefcase } from 'lucide-react';


// Generic API Request Function using Axios and Proxy
async function requestApi<R = any, D = any>(
  config: Omit<AxiosRequestConfig<D>, 'baseURL' | 'url'> & { targetPath: string }
): Promise<ApiResponse<R>> {
  const { targetPath, method = 'GET', data: payloadData, headers: additionalHeaders, params: queryParams, ...axiosClientConfig } = config;
  
  const isServerContext = typeof window === 'undefined';
  let requester: typeof axios | typeof apiClient; // apiClient is for client-side, global axios for server-side
  let finalAxiosConfig: AxiosRequestConfig<D>;

  const requestHeaders: Record<string, string> = {
    'Accept': 'application/json',
    ...(additionalHeaders || {}),
  };
   if (payloadData && method?.toUpperCase() !== 'GET' && !requestHeaders['Content-Type']) {
    requestHeaders['Content-Type'] = 'application/json';
  }
  
  // Add Authorization header if on client and token exists
  if (!isServerContext) {
    const token = localStorage.getItem('token');
    if (token) {
      requestHeaders['Authorization'] = `Bearer ${token}`;
    }
  }
  // Server-side: Direct call to the actual API
  if (isServerContext) {
    requester = axios; // Use global axios instance for direct calls
    const effectiveUrl = `${API_BASE_URL}${targetPath}`;
    // console.log(`[requestApi SERVER_CONTEXT] Direct call to: ${effectiveUrl} with method: ${method}`);
    finalAxiosConfig = {
      method: method as any,
      url: effectiveUrl,
      headers: requestHeaders,
      data: payloadData,
      params: queryParams,
      ...axiosClientConfig,
    };
  } else {
    // Client-side: Call through the Next.js proxy
    requester = apiClient; // Use configured apiClient instance for client-side proxy calls
    // let effectiveProxyUrl: string; // Not needed, baseURL handles it

    if (method?.toUpperCase() === 'GET') {
      // For GET requests, the apiClient's baseURL is /api/proxy.
      // The target path and queryParams will be sent as query parameters to the proxy.
      const proxyParams = new URLSearchParams(queryParams || {});
      proxyParams.set('target', targetPath);
      
      finalAxiosConfig = {
        method: 'GET',
        url: '', // URL is relative to apiClient's baseURL ('/api/proxy')
        headers: requestHeaders, 
        params: proxyParams, // Send target and original query params as query params
        ...axiosClientConfig,
      };
    } else {
      // For POST, PUT, etc., the apiClient's baseURL is /api/proxy.
      // The target path and payloadData are sent in the body to the proxy.
      finalAxiosConfig = {
        method: method as any,
        url: '', // URL is relative to apiClient's baseURL ('/api/proxy')
        data: { target: targetPath, payload: payloadData, queryParamsIfAny: queryParams }, // Proxy expects this structure
        headers: requestHeaders,
        ...axiosClientConfig,
      };
    }
    // The apiClient's request interceptor will handle adding the Authorization header if token exists
  }

  try {
    const response = await requester.request<R>(finalAxiosConfig);
    // Check if the actual API (not the proxy) returned an error within a successful proxy response
    // This depends on how your proxy forwards errors. Assuming proxy forwards status and data directly.
    if (response.data && typeof response.data === 'object' && 'success' in response.data && !(response.data as any).success) {
        const apiErrorData = response.data as any;
        const errorMessage = apiErrorData.message || apiErrorData.error || `API error for ${targetPath}`;
        const logFunc = response.status === 404 ? console.warn : console.error;
        logFunc(`requestApi: Target API error for ${targetPath} (but request was successful):`, errorMessage, "Status (from response body):", response.status, "Response:", response.data);
        return { 
            success: false, 
            data: null,
            error: errorMessage, 
            statusCode: response.status 
        };
    }
    return { success: true, data: response.data, error: null, statusCode: response.status };
  } catch (error) {
    const axiosError = error as AxiosError<any>;
    let errorMessage: string;
    let statusCode: number | null = null;

    if (axiosError.response) { // Error response from server (either target API or proxy)
      statusCode = axiosError.response.status;
      const responseBody = axiosError.response.data;
      if (typeof responseBody === 'string') {
        errorMessage = responseBody;
      } else if (responseBody && typeof responseBody.message === 'string') {
        errorMessage = responseBody.message;
      } else if (responseBody && typeof responseBody.error === 'string') {
        errorMessage = responseBody.error;
      } else if (responseBody && typeof responseBody.errors === 'object' && responseBody.errors !== null) {
        const firstErrorKey = Object.keys(responseBody.errors)[0];
        const firstErrorMsg = responseBody.errors[firstErrorKey]?.[0];
        errorMessage = firstErrorMsg || `Validation error for ${targetPath}. Status: ${statusCode}`;
      } else {
        errorMessage = axiosError.message || `API request to target ${targetPath} failed with status ${statusCode}`;
      }
      const logFunc = statusCode === 404 ? console.warn : console.error;
      logFunc(`requestApi error for ${isServerContext ? 'direct call to' : 'proxy target'} '${targetPath}':`, errorMessage, 'Status Code:', statusCode, 'Original Error:', axiosError);
    } else if (axiosError.request) { // Network error (no response received from proxy or target)
      errorMessage = `No response received from ${isServerContext ? 'target API' : `proxy for target ${targetPath}`}. Check network or server status.`;
       console.error(`requestApi network error for ${isServerContext ? 'direct call to' : `proxy target ${targetPath}`}:`, errorMessage, axiosError);
       statusCode = null; // No HTTP status code from server
    } else { // Other errors (e.g., request setup)
      errorMessage = `An unexpected error occurred making API request via proxy to target ${targetPath}: ${axiosError.message}`;
      console.error(`requestApi unexpected error for proxy target ${targetPath}:`, errorMessage, axiosError);
    }
    return { success: false, data: null, error: errorMessage, statusCode: statusCode };
  }
}


// --- App Settings ---
function transformAppSettingsApiItem(apiItem: AppSettingsApiItem): AppSettings {
  return {
    ...apiItem,
    // Ensure URLs are complete if they aren't already
    logo_url: apiItem.logo_url || (apiItem.logo ? `${API_BASE_URL.replace('/api','')}${apiItem.logo}` : undefined),
    favicon_url: apiItem.favicon_url || (apiItem.favicon ? `${API_BASE_URL.replace('/api','')}${apiItem.favicon}` : undefined),
  };
}

export async function getAppSettings(): Promise<ApiResponse<AppSettings | null>> {
  const response = await requestApi<AppSettingsApiResponse>({ targetPath: '/blog/settings', method: 'GET' });

  if (!response.success || !response.data || !response.data.data || response.data.data.length === 0) {
    let errorMsg = response.error || "No app settings data found from API.";
    if (response.statusCode === 404 && !response.error) {
      errorMsg = "App settings endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
      errorMsg = "Network error fetching app settings. Check connection.";
    }
    console.warn(`getAppSettings: API call failed or no settings data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: null, error: errorMsg, statusCode: response.statusCode };
  }

  try {
    const transformedData = transformAppSettingsApiItem(response.data.data[0]);
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getAppSettings: Transformation error", transformError, "Original data:", response.data.data[0]);
    return { success: false, data: null, error: `Failed to process app settings: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Education ---
function transformEducationApiItem(apiItem: EducationApiItem): EducationEntry {
  let startDate = '';
  if (apiItem.start_date) {
    try {
      startDate = new Date(apiItem.start_date).getFullYear().toString();
    } catch (e) { console.warn(`Invalid start_date for education item ${apiItem.id}: ${apiItem.start_date}`); }
  }
  
  let endDate = '';
  if (apiItem.end_date) {
     try {
      endDate = new Date(apiItem.end_date).getFullYear().toString();
    } catch (e) { console.warn(`Invalid end_date for education item ${apiItem.id}: ${apiItem.end_date}`); }
  }
  const dates = (startDate && endDate) ? `${startDate} – ${endDate}` : (startDate || endDate || 'Date N/A');
  
  return {
    id: apiItem.id,
    item: apiItem.degree_or_certificate,
    institution: apiItem.institution_name,
    dates: dates,
    icon: GraduationCap, // Default icon, as icon_name is not in API
    type: 'education', 
    description: apiItem.description || undefined,
    fieldOfStudy: apiItem.field_of_study || undefined,
    gradeOrScore: apiItem.grade_or_score || undefined,
  };
}
export async function getEducationItems(): Promise<ApiResponse<EducationEntry[]>> {
  const response = await requestApi<EducationApiResponse>({ targetPath: '/portfolio/education-items', method: 'GET' });
  
  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No education data from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Education items endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching education items. Check connection.";
    }
    console.warn(`getEducationItems: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformEducationApiItem);
    if (transformedData.length === 0 && response.success) {
      return { success: true, data: [], error: "No education items available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getEducationItems: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process education: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Projects ---
function transformProjectApiItem(apiItem: ProjectApiItem): Project {
  return {
    id: apiItem.id,
    title: apiItem.name,
    description: apiItem.description,
    url: apiItem.project_url || undefined,
    image: apiItem.image_url || `https://placehold.co/400x300.png?text=${encodeURIComponent(apiItem.name)}`,
    dataAiHint: apiItem.photo_alt_text || (apiItem.project_url ? `website screenshot ${apiItem.name.toLowerCase().split(" ")[0]}` : `abstract ${apiItem.name.toLowerCase().split(" ")[0]}`),
    tech: apiItem.technologies?.map(t => ({
      name: t.name,
      icon: getIcon(t.icon_name),
    })) || [],
  };
}
export async function getProjectsFromApi(): Promise<ApiResponse<Project[]>> {
  const response = await requestApi<ProjectsApiResponse>({ targetPath: '/portfolio/projects', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No projects data from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Projects endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching projects. Check connection.";
    }
    console.warn(`getProjectsFromApi: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }

  try {
    const transformedData = response.data.data.map(transformProjectApiItem);
    if (transformedData.length === 0 && response.success) {
      return { success: true, data: [], error: "No projects available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getProjectsFromApi: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process projects: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Tech Stack ---
function transformTechStackCategoryApiItem(apiItem: TechCategoryApiItem): TechStackCategory {
  return {
    category: apiItem.name,
    items: apiItem.tech_items?.map(item => ({
      name: item.name,
      icon: getIcon(item.icon_url_or_svg),
    })) || [],
  };
}
export async function getTechStack(): Promise<ApiResponse<TechStackCategory[]>> {
  const response = await requestApi<TechStackApiResponse>({ targetPath: '/portfolio/tech-categories', method: 'GET' });
  
  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No tech stack data from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Tech categories endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching tech stack. Check connection.";
    }
    console.warn(`getTechStack: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformTechStackCategoryApiItem);
    if (transformedData.length === 0 && response.success) {
      return { success: true, data: [], error: "No skills details available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getTechStack: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process skills: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Experience ---
function transformExperienceApiItem(apiItem: ExperienceApiItem): ExperienceEntry {
    let startDate = 'N/A';
    if (apiItem.start_date) {
      try {
        startDate = new Date(apiItem.start_date).toLocaleDateString('en-US', { year: 'numeric', month: 'short' });
      } catch (e) { console.warn(`Invalid start_date for experience item ${apiItem.id}: ${apiItem.start_date}`); }
    }
    
    let endDate = 'N/A';
    if(apiItem.is_current) {
        endDate = 'Present';
    } else if (apiItem.end_date) {
       try {
        endDate = new Date(apiItem.end_date).toLocaleDateString('en-US', { year: 'numeric', month: 'short' });
      } catch (e) { console.warn(`Invalid end_date for experience item ${apiItem.id}: ${apiItem.end_date}`); }
    }
    const dates = `${startDate} - ${endDate}`;

  return {
    role: apiItem.title,
    company: apiItem.company_name,
    dates: dates,
    description: apiItem.description_points?.map(p => p.point_text) || [],
    icon: Briefcase, // Default icon
    company_website: apiItem.company_website || undefined,
    location: apiItem.location || undefined,
    short_description: apiItem.short_description || undefined,
  };
}
export async function getExperience(): Promise<ApiResponse<ExperienceEntry[]>> {
  const response = await requestApi<ExperienceApiResponse>({ targetPath: '/portfolio/experiences', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No experience data from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Experience endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching experience. Check connection.";
    }
    console.warn(`getExperience: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformExperienceApiItem);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No experience details available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getExperience: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process experience: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Testimonials ---
function transformTestimonialApiItem(apiItem: TestimonialApiItem): Testimonial {
  return {
    quote: apiItem.quote,
    author: apiItem.author,
    title: apiItem.title,
    image: apiItem.image_url || `https://placehold.co/100x100.png?text=${encodeURIComponent(apiItem.author?.charAt(0) || 'T')}`,
    rating: apiItem.rating || 0,
    dataAiHint: apiItem.data_ai_hint || 'person photo',
  };
}
export async function getTestimonials(): Promise<ApiResponse<Testimonial[]>> {
  const response = await requestApi<TestimonialsApiResponse>({ targetPath: '/portfolio/testimonials', method: 'GET' });
  
  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No testimonials data from API.";
     if (response.statusCode === 404 && !response.error) {
        errorMsg = "Testimonials endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching testimonials. Check connection.";
    }
    console.warn(`getTestimonials: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformTestimonialApiItem);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No testimonials available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getTestimonials: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process testimonials: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Patreon Tiers ---
function transformPatreonTierApiItem(apiItem: PatreonTierApiItem): PatreonTier {
  return {
    name: apiItem.name,
    price: apiItem.price,
    benefits: apiItem.benefits,
    icon: getIcon(apiItem.icon_name),
    linkUrl: apiItem.link_url || '#',
  };
}
export async function getPatreonTiers(): Promise<ApiResponse<PatreonTier[]>> {
  const response = await requestApi<PatreonTiersApiResponse>({ targetPath: '/portfolio/patreon-tiers', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No Patreon tiers data from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Patreon tiers endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching Patreon tiers. Check connection.";
    }
    console.warn(`getPatreonTiers: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformPatreonTierApiItem);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No support tiers available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getPatreonTiers: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process support tiers: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Interests ---
function transformInterestApiItem(apiItem: InterestApiItem): Interest {
  return {
    name: apiItem.name,
    icon: getIcon(apiItem.icon_name),
    description: apiItem.description || '',
  };
}
export async function getInterests(): Promise<ApiResponse<Interest[]>> {
  const response = await requestApi<InterestsApiResponse>({ targetPath: '/portfolio/interests', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No interests data from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Interests endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching interests. Check connection.";
    }
    console.warn(`getInterests: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformInterestApiItem);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No interests available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getInterests: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process interests data: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Navigation Links ---
function transformNavigationLinkApiItem(apiItem: NavigationLinkApiItem): NavigationLink {
  return {
    name: apiItem.name,
    href: apiItem.url,
    iconName: apiItem.icon_name || 'LinkIcon', // Default if not provided
  };
}
export async function getNavigationLinks(): Promise<ApiResponse<NavigationLink[]>> {
  const response = await requestApi<NavigationLinksApiResponse>({ targetPath: '/portfolio/navigation-links', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No navigation links from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Navigation links endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching navigation links. Check connection.";
    }
    console.warn(`getNavigationLinks: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode }; // Return empty array on failure
  }
  
  try {
    const transformedData = response.data.data.map(transformNavigationLinkApiItem);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No navigation links available from API.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getNavigationLinks: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process navigation links: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Social Links ---
function transformSocialLinkApiItem(apiItem: SocialLinkApiItem): SocialLink {
  return {
    name: apiItem.name,
    url: apiItem.url,
    iconName: apiItem.icon_name || 'LinkIcon', // Default if not provided
  };
}
export async function getSocialLinks(): Promise<ApiResponse<SocialLink[]>> {
  const response = await requestApi<SocialLinksApiResponse>({ targetPath: '/portfolio/social-links', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No social links from API.";
     if (response.statusCode === 404 && !response.error) {
        errorMsg = "Social links endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching social links. Check connection.";
    }
    console.warn(`getSocialLinks: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode }; // Return empty array on failure
  }
  
  try {
    const transformedData = response.data.data.map(transformSocialLinkApiItem);
     if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No social links available from API.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getSocialLinks: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process social links: ${transformError}`, statusCode: response.statusCode };
  }
}


// --- Blog Posts ---
interface GetBlogPostsFilters {
  category?: string;
  tag?: string;
  limit?: number;
  page?: number;
}

export function transformPostResourceToBlogPost(postResource: PostResource): BlogPost {
  // Ensure seo_detail and its keywords are handled safely
  let keywordsString = "";
  if (postResource.seo_detail?.keywords) {
    keywordsString = Array.isArray(postResource.seo_detail.keywords)
      ? postResource.seo_detail.keywords.join(', ')
      : postResource.seo_detail.keywords;
  } else if (postResource.tags && postResource.tags.length > 0) {
    keywordsString = postResource.tags.map(tag => tag.name).join(', ');
  }

  return {
    id: postResource.id,
    slug: postResource.slug,
    frontmatter: {
      id: postResource.id,
      title: postResource.title,
      date: postResource.published_at, 
      category: postResource.categories?.[0]?.name || 'Uncategorized',
      excerpt: postResource.sub_title || postResource.body?.substring(0, 150) + '...' || "No excerpt available.",
      author: postResource.user?.name || personalInfo.name,
      image: postResource.cover_photo_url || `https://placehold.co/1200x600.png?text=${encodeURIComponent(postResource.title)}`,
      tags: postResource.tags?.map(tag => tag.name) || [],
      dataAiHint: postResource.photo_alt_text || `blog post ${postResource.title?.toLowerCase().split(" ")[0] || 'image'}`,
      featured: postResource.is_featured || false,
      showTopAd: postResource.show_top_ad || false,
      showBottomAd: postResource.show_bottom_ad || false,
      affiliateProducts: postResource.affiliate_products || [], 
      seo_detail: postResource.seo_detail || null,
      likes: postResource.likes || 0,
      comments_count: postResource.comments_count || 0,
      views_count: postResource.views_count || 0,
    },
    content: postResource.body || "", 
    comments: postResource.comments || [],
  };
}

export async function getBlogPosts(filters?: GetBlogPostsFilters): Promise<ApiResponse<BlogPost[]>> {
  const queryParams: Record<string, string | number> = {};
  if (filters?.category) queryParams['category'] = filters.category;
  if (filters?.tag) queryParams['tag'] = filters.tag;
  if (filters?.limit) queryParams['limit'] = filters.limit;
  if (filters?.page) queryParams['page'] = filters.page;
  
  const response = await requestApi<PostsApiResponse>({ 
    targetPath: '/blog/posts', 
    method: 'GET',
    params: queryParams 
  });
  
  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No blog posts from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = `Blog posts not found (404). Filters: ${JSON.stringify(filters)}`;
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching blog posts. Check connection.";
    }
    const logFunc = response.statusCode === 404 ? console.warn : console.error;
    logFunc(`getBlogPosts: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}. Filters: ${JSON.stringify(filters)}`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformPostResourceToBlogPost);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No blog posts found matching criteria.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error(`getBlogPosts: Transformation error. Filters: ${JSON.stringify(filters)}`, transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process blog posts: ${transformError}`, statusCode: response.statusCode };
  }
}

export async function getBlogPostBySlug(slug: string): Promise<ApiResponse<BlogPost | null>> {
  const response = await requestApi<SinglePostApiResponse>({ targetPath: `/blog/posts/slug/${slug}`, method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || `No blog post data found for slug "${slug}".`;
    if (response.statusCode === 404 && !response.error) {
        errorMsg = `Blog post "${slug}" not found (404).`;
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = `Network error fetching blog post "${slug}". Check connection.`;
    }
    const logFunc = response.statusCode === 404 ? console.warn : console.error;
    logFunc(`getBlogPostBySlug (slug: ${slug}): API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: null, error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = transformPostResourceToBlogPost(response.data.data);
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error(`getBlogPostBySlug (slug: ${slug}): Transformation error`, transformError, "Original data:", response.data.data);
    return { success: false, data: null, error: `Failed to process blog post "${slug}": ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Blog Categories ---
function transformBlogCategoryApiItem(apiItem: CategoryApiItem): BlogCategory {
  return {
    id: apiItem.id,
    name: apiItem.name,
    slug: apiItem.slug,
    count: apiItem.posts_count || 0,
  };
}
export async function getBlogCategoriesFromApi(): Promise<ApiResponse<BlogCategory[]>> {
  const response = await requestApi<CategoriesApiResponse>({ targetPath: '/blog/categories', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No blog categories from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Blog categories endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching blog categories. Check connection.";
    }
    const logFunc = response.statusCode === 404 ? console.warn : console.error;
    logFunc(`getBlogCategoriesFromApi: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }
  
  try {
    const transformedData = response.data.data.map(transformBlogCategoryApiItem);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No blog categories available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getBlogCategoriesFromApi: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process blog categories: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Blog Tags ---
function transformTagResourceToBlogTag(tagResource: TagResource): BlogTag {
  return {
    id: tagResource.id,
    name: tagResource.name,
    slug: tagResource.slug,
    count: tagResource.posts_count || 0,
  };
}
export async function getBlogTagsFromApi(): Promise<ApiResponse<BlogTag[]>> {
  const response = await requestApi<TagsApiResponse>({ targetPath: '/blog/tags', method: 'GET' });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "No blog tags from API.";
    if (response.statusCode === 404 && !response.error) {
        errorMsg = "Blog tags endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error fetching blog tags. Check connection.";
    }
    // Keep as warn because BlogLayout has a fallback to derive tags from posts
    console.warn(`getBlogTagsFromApi: API call failed or no data. Error: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: [], error: errorMsg, statusCode: response.statusCode };
  }

  try {
    const transformedData = response.data.data.map(transformTagResourceToBlogTag);
    if (transformedData.length === 0 && response.success) {
        return { success: true, data: [], error: "No blog tags available.", statusCode: response.statusCode };
    }
    return { success: true, data: transformedData, error: null, statusCode: response.statusCode };
  } catch (e) {
    const transformError = e instanceof Error ? e.message : 'Transformation error';
    console.error("getBlogTagsFromApi: Transformation error", transformError, "Original data:", response.data.data);
    return { success: false, data: [], error: `Failed to process blog tags: ${transformError}`, statusCode: response.statusCode };
  }
}

// --- Contact Form ---
export async function submitContactForm(payload: ContactFormPayload): Promise<ApiResponse<ContactSubmissionResource>> {
  const response = await requestApi<{ data: ContactSubmissionResource }, ContactFormPayload>({
    targetPath: '/contact-submissions',
    method: 'POST',
    data: payload,
  });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "Failed to submit contact form or no submission data in response.";
    if (response.statusCode === 422 && !response.error) { 
        errorMsg = `Validation failed: ${response.error || "Please check your input."}`;
    } else if (response.statusCode === 404 && !response.error) {
        errorMsg = "Contact submission endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error submitting contact form. Check connection.";
    }
    console.error(`submitContactForm: API error. Msg: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: null, error: errorMsg, statusCode: response.statusCode };
  }
  
  const successMessage = (response.data as any).message || (response.data as any).error || "Your message has been successfully submitted."; 
  return { success: true, data: response.data.data, error: successMessage, statusCode: response.statusCode };
}

// --- Blog Comments ---
export async function submitBlogPostComment(postId: string | number, payload: NewCommentPayload): Promise<ApiResponse<CommentResource>> {
  const response = await requestApi<SubmitCommentApiResponse, NewCommentPayload>({
    targetPath: `/blog/posts/${postId}/comments`,
    method: 'POST',
    data: payload,
  });

  if (!response.success || !response.data || !response.data.data) {
    let errorMsg = response.error || "Failed to submit comment.";
    if (response.statusCode === 422 && !response.error) {
      errorMsg = `Validation failed: ${response.error || "Please check your comment."}`;
    } else if (response.statusCode === 404 && !response.error) {
      errorMsg = "Comment submission endpoint not found (404).";
    } else if (response.statusCode === null && response.error?.includes("Failed to fetch")) {
        errorMsg = "Network error submitting comment. Check connection.";
    }
    console.error(`submitBlogPostComment: API error for post ${postId}. Msg: "${errorMsg}". Status: ${response.statusCode}.`);
    return { success: false, data: null, error: errorMsg, statusCode: response.statusCode };
  }
  const successMessage = response.data.message || "Comment submitted successfully.";
  return { success: true, data: response.data.data, error: successMessage, statusCode: response.statusCode };
}


// --- Authentication Stubs ---
export async function loginUser(payload: LoginPayload): Promise<ApiResponse<AuthResponse>> {
  console.log("Attempting to log in user via API with payload:", payload);
  // Placeholder: Replace with actual API call using requestApi
  // const response = await requestApi<AuthResponse, LoginPayload>({
  //   targetPath: '/auth/login', // Replace with your actual login endpoint
  //   method: 'POST',
  //   data: payload,
  // });
  // return response;
  
  await new Promise(resolve => setTimeout(resolve, 1000));
   return { success: false, data: null, error: "Login functionality not yet implemented with API.", statusCode: 501 };
}

export async function registerUser(payload: RegisterPayload): Promise<ApiResponse<AuthResponse>> {
  console.log("Attempting to register user via API with payload:", payload);
  // Placeholder: Replace with actual API call using requestApi
  // const response = await requestApi<AuthResponse, RegisterPayload>({
  //   targetPath: '/auth/register', // Replace with your actual register endpoint
  //   method: 'POST',
  //   data: payload,
  // });
  // return response;

  await new Promise(resolve => setTimeout(resolve, 1000));
  return { success: false, data: null, error: "Registration functionality not yet implemented with API.", statusCode: 501 };
}

export async function logoutUser(): Promise<ApiResponse<null>> {
  console.log("Attempting to log out user via API");
  // Placeholder: Replace with actual API call using requestApi
  // const response = await requestApi<null, null>({
  //   targetPath: '/auth/logout', // Replace with your actual logout endpoint
  //   method: 'POST', 
  // });
  // return response;

  await new Promise(resolve => setTimeout(resolve, 500));
  return { success: true, data: null, error: null, statusCode: 200 }; // Simulate success for UI
}

// Axios instance for client-side calls (to be routed via proxy)
const apiClient = axios.create({
  baseURL: PROXY_PATH, // All client-side requests go to our proxy
  timeout: 10000,
  headers: {
    'Content-Type': 'application/json',
    Accept: 'application/json',
  },
});

// Client-side request interceptor for apiClient (e.g., adding auth token)
apiClient.interceptors.request.use(
  (config: InternalAxiosRequestConfig) => { // Use InternalAxiosRequestConfig
    if (typeof window !== 'undefined') { // Ensure localStorage is available
      const token = localStorage.getItem('token');
      if (token) {
        config.headers = config.headers || {};
        config.headers.Authorization = `Bearer ${token}`;
      }
    }
    return config;
  },
  (error) => {
    console.error('ApiClient Request Error Interceptor:', error);
    return Promise.reject(error);
  }
);

// Client-side response interceptor for apiClient (e.g., global error handling)
apiClient.interceptors.response.use(
  (response: AxiosResponse) => response,
  (error: AxiosError) => {
    let targetPathDisplay = 'N/A';
    if (error.config) {
      if (error.config.method?.toUpperCase() === 'GET' && error.config.params) {
        // For GET requests to the proxy, target is in params (which should be a URLSearchParams object)
        if (typeof error.config.params.get === 'function') {
          targetPathDisplay = error.config.params.get('target') || 'N/A (target param missing)';
        } else {
          // Fallback if params is not URLSearchParams (less likely for our setup)
          targetPathDisplay = (error.config.params as any).target || 'N/A (target not in URLSearchParams-like params)';
        }
      } else if (error.config.data) {
        // For POST/PUT etc. to the proxy, target might be in body
        try {
          const requestData = typeof error.config.data === 'string' ? JSON.parse(error.config.data) : error.config.data;
          if (requestData && typeof requestData.target === 'string') {
            targetPathDisplay = requestData.target;
          }
        } catch (e) {
          targetPathDisplay = 'N/A (could not parse request body for target)';
        }
      }
    }

    const message =
      (error.response?.data as any)?.message || error.message || 'Unknown error occurred via apiClient';

    console.error(
      `ApiClient API Error Interceptor:\n` +
      `  Proxy URL: ${apiClient.defaults.baseURL || PROXY_PATH}${error.config?.url || ''}\n` +
      `  Target API Path: ${targetPathDisplay}\n` +
      `  Method: ${error.config?.method?.toUpperCase() || 'N/A'}\n` +
      `  Status: ${error.response?.status || 'N/A'}\n` +
      `  Message: ${message}\n` +
      `  Response Data: ${error.response?.data ? JSON.stringify(error.response.data, null, 2) : 'N/A'}`
    );
    return Promise.reject(error);
  }
);

